#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# USAGE: MSGx.py -t YYYYmmDDHHMM [-sat n -src <sloc> -dst <dloc> -nat]
#           where DD=01..31, HH=00..23, MM=00[,05,10],15[,20,25],30[,35,40],45[,50,55] 
#                 n=1,2,3,4,RSS,IODC
#                 sloc=location source files
#                 dloc=location generated file
#                 -nat: use zipped native format
#************************************************************************
#
# European geostationary satellite Meteosat-x (MSGx) at 0.0, 9.5, 45.5 degrees East
#------------------------------------------------------------------------
# EUMETSAT is EUMETCast disseminating MSGx data in EUMETSAT_Data_Channel_y
# Meteosat-11 (MSG4), at 0.0 noon is exactly 12:00 UTC, instrument 'seviri'
#
# Channel | VIS006 |   HRV  | VIS008 | IR_016 | IR_039 | WV_062 |
# WaveLen |  0.635 |   0.7  |   0.81 |   1.64 |   3.90 |   6.25 |
# --------+--------+--------+--------+--------+--------+--------+
# Channel | WV_073 | IR_087 | IR_097 | IR_108 | IR_120 | IR_134 |
# WaveLen |   7.35 |   8.70 |   9.66 |  10.80 |  12.00 |  13.40 |
#
# CH-3123 Belp     , 2021/01/17       License GPL3          (c) Ernst Lobsiger
#         Hilversum, 2022/09/20       License GPL3          (c) Rob Alblas
#
#************************************************************************

# I need
import os
import sys
import shutil
import argparse
from os.path import expanduser

# Pytroll/SatPy needs these
from satpy import Scene
from glob import glob

# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

#Choose OS: uncomment one
opsys='Linux'
#opsys='Windows'

#Location where tools and data of satpy are located
home=expanduser("~")
toodrp = home+'/tools/satpy'
datdrp = home+'/tools/satpy'

def get_all_areas(sel=''):
  from satpy.resample import get_area_file
  areafiles=get_area_file()
  print('Areas:')
  for fs in areafiles:
    print('  '+fs+':')
    with open(fs) as f:
      for line in f:
        ln=line.strip('\n')
        if ln[len(ln)-1:] == ':' and ln[0]!=' ':
          ln=ln.strip(':')
          if sel=='' or sel in ln:
            print('    '+ln)

def is_segdir(rchan,Date):
  Yea=Date[:4]
  Mon=Date[4:6]
  Day=Date[6:8]
  return '/srv/rec_0/'+rchan+'/' + Yea + '/' + Mon + '/' + Day

# if segdir/<y>/<m>/<d> exists: return this, otherwise return segdir
def mkloc_ymd(segdir,y,m,d):
  osegdir=segdir+'/'+y+'/'+m+'/'+d
  if os.path.isdir(osegdir):
    return osegdir
  return segdir

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

if (opsys=='Windows'):
  convert_cmd=toodrp + '/EMCtools/exefiles/convert.exe'
  remove_cmd='del'
  bckclr='rgb(245,245,245)'
else:
  from shutil import which
  if which('magick'):
    convert_cmd='magick'
  elif which('convert'):
    convert_cmd='convert'
  else:
    sys.exit("ERROR: 'magick' or 'convert' not found.")

  remove_cmd='rm -f'
  bckclr='rgb\(245,245,245\)'

# Needed for grid labels that have their own color setting (see above grid definition)
# This did segfault for a while and has then been fixed by Dave (may or may not work)!
# import aggdraw

# Why to hell is it not working?
#from satpy.utils import debug_on
#debug_on()

# Command line parsing
usage1 = 'Usage: ' + sys.argv[0] + ' -t YYYYmmDDHHMM  [-sat RSS|IOC|1|2|3|4] [-src sourcedir] [-dst destinationdir] [-area <area>] [-composite <composite>]'

parser = argparse.ArgumentParser('MSGx')
parser.add_argument('-t'        ,default=''      , help="time: YYYYmmDDHHMM")
parser.add_argument('-sat'      ,default='FULL'  , help="RSS|IODC|FULL|1|2|3|4]")
parser.add_argument('-src'      ,default=''      , help="source dir., MUST be absolute path!")
parser.add_argument('-nat'      ,default='',nargs='?', const='native',help="use native format")
parser.add_argument('-dst'      ,default=''      , help="destination dir.")
parser.add_argument('-o'        ,default=''      , help="output file")
parser.add_argument('-area'     ,default=''      , help="area, 'list' gives a list, 'list=<str>': items containing <str>")
parser.add_argument('-composite',default=''      , help="composite, 'list' gives a list")
parser.add_argument('-v'        ,default='noview',nargs='?',const='view', help="view")

args = parser.parse_args()
Dat=args.t
asat=args.sat
src=args.src
filetype=args.nat
dst=args.dst
ofile=args.o
area=args.area
composite=args.composite
view=args.v
if opsys=='Windows':
  viewer='eog'
else:
  viewer='eog'

if (area[:4]=='list'):
  get_all_areas(area[5:])
  sys.exit()

if Dat=='':
  sys.exit(usage1)

if ((asat=='1') | (asat=='2') | (asat=='3') | (asat=='4')):
  satid='MSG'+asat
  sat1=satid
  sat2='Meteosat-'+str(7+int(asat))
  rchan='E1B-GEO-2'
else:
  if (asat=='FULL'):
    satid='MSGx_'
    sat1='MSG*_____'
    sat2='Meteosat-x'
    rchan='E1B-GEO-2'
  else:
    if ((asat=='RSS') | (asat=='IODC')):
      satid='MSGx_'+asat
      sat1='MSG*_'+asat
      sat2='Meteosat-x'
      if (asat=='RSS'):
        rchan='E1B-GEO-5'
      else:
        rchan='E1B-GEO-1'
    else:
      sys.exit(usage1)

tsrvc='Basic'

if filetype=='native':
  rectype='internet'
  rchan='Data Store'
else:
  rectype='SR-1'

#satid=MSG1      |MSG2      |MSG3      |MSG4      | MSGx_RSS | MSGx_IODC 
#sat1 =MSG1      |MSG2      |MSG3      |MSG4      | MSG*_RSS | MSG*_IODC 
#sat2 =Meteosat-1|Meteosat-2|Meteosat-3|Meteosat-4|Meteosat-x|Meteosat-x
sat=sat1

# Get substrings
Yea=Dat[:4]
Mon=Dat[4:6]
Day=Dat[6:8]
Hou=Dat[8:10]
Min=Dat[10:]


if (src==''):
  segdir=is_segdir(rchan,Dat)
else:
  segdir = src

if (segdir[len(segdir)-1]=='/'):
  segdir = segdir[:-1]

# try to search for YYYY/mm/DD location, extend segdir if exists 
#   (asume files are saved that way)
segdir=mkloc_ymd(segdir,Yea,Mon,Day)

print("Source dir.: " + segdir)

curdir=os.getcwd()

# Available composites in this script (far more in ../satpy/etc/composites/seviri.yaml):
# 'overview', 'natural_color', 'realistic_colors', 'hrv_clouds', 'night_fog', 'ir108_3d'
#composite = 'overview'


# Choose your area (remember that HRV data has only limited coverage) If not specified: auto-area
if (area == 'full_scan'):
  area=''

if (area == ''):
  if (asat=='IODC'):
    area = 'msg_seviri_iodc_3km'
  else:
    if (asat == 'RSS'):
#      area = 'seviri_rss'
      area = 'msg_seviri_fes_3km'
    else:
      area = 'msg_seviri_fes_3km'
#area = 'eurol'
#area = 'westminster'
#area = 'isleofman'
#area = 'switzerland'
#area = 'afghanistan'

if (composite == ''):
  composite = 'natural_color'

print("Area     : " + area)
print("Composite: " + composite)

# Configure your individual overlays, either True/False or 1/0 do work
ADDgrid    =  True
ADDcoasts  =  True
ADDborders =  True
ADDrivers  =  True
ADDpoints  =  True

# OVRCache speeds up static overlays. These are stored in your .../EMCdata/cache
# With caching you will sometimes have to delete *.png OVRcache files manually !
OVRcache   =  False
# GEOcache allows for a resample cache of GEO satellites (speedy massproduction)
# These caches are stored in subdirectories with unique names .../nn_lut-*.zarr
GEOcache   =  False

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^


# Set temporary directory and image directory
tmpdir = datdrp + '/EMCdata/tmpdirs/xmsgx'
if (dst==''):
  imgdir = datdrp + '/EMCdata/images/Meteosat-x'
else:
  imgdir = dst
  if (imgdir[len(imgdir)-1]=='/'):
    imgdir = imgdir[:-1]

if (ofile==''):
  if (view=='view'):
    ofile='satpy_preview.jpg'
  else:
    ofile=satid+'-'+Dat[:8]+'-SLO-'+Dat[8:]+'-'+composite+'-'+area+'.jpg'

# Dir must exist
try:
   os.chdir(tmpdir)
except:
   sys.exit(tmpdir+': Cannot change to tmp directory ...')

# Clean tmp directory
os.system(remove_cmd+' H-000*')

# Output of global_scene.available_composite_names() when all seviri channels are loaded
# **************************************************************************************
# ['airmass', 'ash', 'cloudtop', 'cloudtop_daytime', 'colorized_ir_clouds', 'convection',
# 'day_microphysics', 'day_microphysics_winter', 'dust', 'fog', 'green_snow', 'hrv_clouds',
# 'hrv_fog', 'hrv_severe_storms', 'hrv_severe_storms_masked', 'ir108_3d', 'ir_cloud_day',
# 'ir_overview', 'ir_sandwich', 'natural_color', 'natural_color_nocorr', 'natural_color_raw',
# 'natural_color_with_night_ir', 'natural_color_with_night_ir_hires', 'natural_enh',
# 'natural_with_night_fog', 'night_fog', 'night_ir_alpha', 'night_ir_with_background',
# 'night_ir_with_background_hires', 'night_microphysics', 'overview', 'overview_raw',
# 'realistic_colors', 'snow', 'vis_sharpened_ir']

# We only decompress what is actually needed, see file seviri.yaml
needed_files = {'overview'         : ['VIS006', 'VIS008', 'IR_108'],
                'natural_color'    : ['VIS006', 'VIS008', 'IR_016'],
                'realistic_colors' : ['VIS006', 'VIS008', 'HRV'],
                'hrv_clouds'       : ['IR_108', 'HRV'],
                'night_fog'        : ['IR_039', 'IR_108', 'IR_120', 'IR_134'],
                'ir108_3d'         : ['IR_108'],
                'all'              : ['VIS006', 'VIS008', 'IR_016', 'IR_039',
                                      'WV_062', 'WV_073', 'IR_087', 'IR_097',
                                      'IR_108', 'IR_120', 'IR_134', 'HRV']
               }

# Some composite names are too long for IM annotation left of pic
abbreviation = {'overview'         : 'overview',
                'natural_color'    : 'natural_color',
                'realistic_colors' : 'realist_color',
                'hrv_clouds'       : 'hrv_clouds',
                'night_fog'        : 'night_fog',
                'ir108_3d'         : 'ir108_3d'
               }

#If composite isn't in list of needed files: just take all channels
if composite in needed_files:
  composite_neededfiles=composite
else:
  composite_neededfiles='all'

#If composite isn't in list of abbreviations: just take full name
if composite in abbreviation:
  composite_abbr=abbreviation[composite]
else:
  composite_abbr=composite

# we are now in tmpdir, so uncompress/unzip will put files there.
# If only list of composites is needed: do what's really needed:
#   xrit: one file of each channel, decompress not needed
#   native: unzip
if filetype == 'native':
  files = glob(segdir +'/'+ sat+'*' + Dat + '*.zip')
  for f in files:
    print('unzip -o ' + f)
    os.system('unzip -o ' + f)
    if composite == 'list':    # just unzip 1 file, for composite list
      break
else:
  for b in needed_files[composite_neededfiles]:
    files = glob(segdir + '/H-000-'+sat+'*' + str(b) + '*' + Dat + '*-C_')
    # Decompress all files found
    for f in files:
      if composite == 'list':
#        os.system(toodrp + '/EMCtools/exefiles/xRITDecompress ' + f)
        shutil.copy2(f, tmpdir); # just copy 1 file of each channel, for composite list, decompr. not needed
        break
      else:
        os.system(toodrp + '/EMCtools/exefiles/xRITDecompress ' + f)

# Meteosat-11 sends only 1 PROlogue and 1 EPIlogue file per 15 minute time slot
# But e.g. the hrit_goes reader insisted that there is a separate PRO per band?
if filetype == 'native':
  files = glob(sat+'*' + Dat + '*.nat')
else:
  files = glob(segdir + '/H-000-'+sat+'*PRO*' + Dat + '*')
  for f in files:
      shutil.copy2(f, tmpdir);

  files = glob(segdir + '/H-000-'+sat+'*EPI*' + Dat + '*')
  for f in files:
      shutil.copy2(f, tmpdir);

  # Now all the files needed should be decompressed in tmpdir
  files = glob(tmpdir + '/H-000-'+sat+'*' + Dat + '*')

# We should have got at least some good files
if files == []:
    sys.exit('Sorry, no good files found ...')

if filetype == 'native':
  global_scene = Scene(filenames = files, reader = 'seviri_l1b_native')
else:
  global_scene = Scene(filenames = files, reader = 'seviri_l1b_hrit')

if (composite=='list'):
  print('All composites:')
  composites=global_scene.available_composite_names()
  print(str(composites))
  quit()

global_scene.load([composite])

# This is a speedup by projection/area cache. It works for GEOstationary satellites only. 
if GEOcache:
    geocache = datdrp + '/EMCdata/cache/'+satid
else:
    geocache = None

# Full earth seviri image has South on top, image size is 3712x3712 pixels, resample rotates it (no need for IM)
if area=='swath':
  new_scene = global_scene
else:
  new_scene = global_scene.resample(area, radius_of_influence = 20000, cache_dir = geocache) #, reduce_data = False)

# Next is line by Christian Peters, thanks a lot!
# try:except: added after Graham Woolf's problems
try: # This is for real (multichannel) composites
    layers, height, width = new_scene[composite].shape
except: # In case of a single channel 'composite'
    height, width = new_scene[composite].shape

# Autoscale font_size (fs) of grid labels
fs = int((height + width) / 100)

# my_font = aggdraw.Font('red', datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
#           opacity = 255, size = fs) # aggdraw.Font takes only 4 arguments
# Font color can either be specified 'white' or (R, G, B) = (255, 255, 255)
# .. 'font': my_font} below may or may not work anymore now or in the future.
# If a direct font definition does not work, use the inline definition below:

# New interface to pycoast + POIs have been integrated 2020, thanks Martin !!
my_grid  = {'major_lonlat': (10,10), 'minor_lonlat': (2, 2),
            # Opacity 0 will hide the line, values 0 ... 255  EL
            'outline': (255, 255, 255) , 'outline_opacity': 255,
            'minor_outline': (200, 200, 200),'minor_outline_opacity': 127,
            'width': 1.5, 'minor_width': 1.0, 'minor_is_tick': False,
            'write_text': True, 'lat_placement': 'lr', 'lon_placement': 'b',
            'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf', 'font_size': fs}
            # minor_is_tick: False draws a line not ticks, True does ticks
            # label placement l,r,lr for latitude and t,b,tb for longitude

# Colors are specified as strings (140 HTML color names) or as triple (R, G, B) 8Bit.
# Best 'resolution' is automatically choosen according to area_def, no need to touch.
# Coasts also has keys ..,'fill': 'green', 'fill_opacity': 50} for land mass filling.
# 'level' defaults to 1 but should be choosen at least 2 for rivers (1 = 0 in GSHHG).
my_coasts  = {'outline': (255, 255,   0), 'width': 1.5, 'level': 1} # , 'resolution': 'i'}  # level 1 .. 4
my_borders = {'outline': (255,   0,   0), 'width': 1.0, 'level': 1} # , 'resolution': 'h'}  # level 1 .. 3
my_rivers  = {'outline': (  0,   0, 255), 'width': 1.0, 'level': 3} # , 'resolution': 'f'}  # level 1 .. 11

# List of POIs (Station, Cities, etc. ...)
my_poi_list = [((13.4052, 52.5214), 'Berlin'),
               (( 2.3522, 48.8566), 'Paris'),
               (( 0.1278, 51.5074), 'London'),
               ((-3.1880, 55.9539), 'Edinburgh'),
               ((-6.2598, 53.3514), 'Dublin'),
               (( 7.4989, 46.8918), 'Belp'),
               (( 5.1738, 52.2063), 'Hilversum')]

# You may find other True Type Fonts (*.ttf) in /usr/share/fonts/truetype/...
# Windows 10 PRO users may find more True Type Fonts in C:\Windows\Fonts\ ...
my_points = {'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
             'font_size': 20, 'points_list': my_poi_list, 'symbol': 'circle',
             'ptsize': 20, 'outline': 'black', 'fill': 'red', 'width': 5.0,
             'outline_opacity': 255,'fill_opacity': 255, 'box_outline': 'navy',
             'box_linewidth': 3.0, 'box_fill': 'gold', 'box_opacity': 255}

# Cache is useful for speedup. Overlays are stored as tranparent *.png per satellite!
my_cache = {'file': datdrp + '/EMCdata/cache/' + satid + '/' + composite, 'regenerate': False}

# Setup indivdual overlays dictionary
my_overlays = {}
if ADDgrid:    # Add key grid
    my_overlays['grid'] = my_grid
if ADDcoasts:  # Add key coasts
    my_overlays['coasts'] = my_coasts
if ADDborders: # Add key borders
    my_overlays['borders'] = my_borders
if ADDrivers:  # Add key rivers
    my_overlays['rivers'] = my_rivers
if ADDpoints:  # Add key points
    my_overlays['points'] = my_points
if OVRcache:   # Add key cache
    my_overlays['cache'] = my_cache

# Enhance image with overlays and save *.png to temporary directory
new_scene.save_dataset(composite, satid+'.png',
          overlay = {'coast_dir': datdrp + '/EMCdata/gshhg-shp',
         'overlays' : my_overlays}) # 2^6=32 combinations should do

# ****************************************************************
# NEW POST PROCESSING: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# ****************************************************************

# ImageMagick (IM) postprocessing, added directory logos in EMCdata

# Use IM to annotate (maybe also -equalize) and convert to *.jpg
# www.imagemagick.org/Usage/crop/#splice or .../annotating/#labeling
# This is the syntax used -draw "text OffsetX,OffsetY 'Hello World!'"
# This annotates to the left, numbers depend on original image size
# Colors can be {dark,light}blue, etc. or rgb(R,G,B) with \( escaped

def magick_string(tmpdir,loc_logo,EffHeight,ofile):
    """
    It seems ImageMagick under Windows cannot cope with multiline text
    that includes \n newlines. We have to draw every line separately.
    This scales all pixel values for IM according to the image height.
    Should make it easier to use annotation and logos at the left side.
    This has been optimized using MSG4 full scan with 3712x3712 pixels.
    """

    logo1 = loc_logo + '/'+'eumetsat_200x199.png'
    logo2 = loc_logo + '/PyTROLL_400x400.jpg'

    OptHeight = 3712
    OptSplice = 640
    OptLogo1s = 540
    OptLogo1x = 50   # 40 for 'GNU-Linux-Logo-Penguin-SVG.png'
    OptLogo1y = 40   # 60 for 'GNU-Linux-Logo-Penguin-SVG.png'
    OptLogo2s = 530
    OptLogo2x = 55
    OptLogo2y = 55
    OptPoints = 90

    fac = EffHeight/OptHeight/2
    EffSplice = str(int(fac*OptSplice+0.5))
    EffLogo1s = str(int(fac*OptLogo1s+0.5))
    EffLogo1x = str(int(fac*OptLogo1x+0.5))
    EffLogo1y = str(int(fac*OptLogo1y+0.5))
    EffLogo2s = str(int(fac*OptLogo2s+0.5))
    EffLogo2x = str(int(fac*OptLogo2x+0.5))
    EffLogo2y = str(int(fac*OptLogo2y+0.5))
    EffPoints = str(int(fac*OptPoints+0.5))

    satname=satid
    sensor= 'seviri'

    legend = [' \'DATE:\'\" '     , ' \''+Yea+'/'+Mon+'/'+Day+'\'\" ',
              ' \'TIME:\'\" '     , ' \''+Hou+':'+Min+' UTC'+'\'\" ',
              ' \'SOURCE:\'\" '   , ' \'EUMETCast\'\" ',
              ' \'SERVICE:\'\" '  , ' \''+tsrvc+'\'\" ',
              ' \'CHANNEL:\'\" '  , ' \''+rchan+'\'\" ',
              ' \'RECEIVER:\'\" ' , ' \''+rectype+'\'\" ',
              ' \'SATELLITE:\'\" ', ' \''+satname+'\'\" ',
              ' \'SENSOR:\'\" '   , ' \''+sensor+'\'\" ',
              ' \'COMPOSITE:\'\" ', ' \''+composite_abbr+'\'\" ',
              ' \'AREA:\'\" '     , ' \''+area+'\'\" ']
    multidraw = ''
    yoff = fac*470
    for n in range(0, len(legend)):
        if (n % 2) == 0:
            xoff = fac*40
            yoff += fac*150
        else:
            xoff = fac*60
            yoff += fac*100
        multidraw += '-draw \"text +'+str(int(xoff+0.5))+'+'+str(int(yoff+0.5))+legend[n]

    ifile=tmpdir+'/'+satid+'.png'
    cmdstr = convert_cmd+' '+ifile+' '\
    + '-font Nimbus-Sans '\
    + '-gravity west -background '+ bckclr +' '\
    + '-splice '+EffSplice+'x0 '\
    + logo1+' -gravity northwest -geometry '+EffLogo1s+'x'\
    + EffLogo1s+'+'+EffLogo1x+'+'+EffLogo1y+' -composite '\
    + logo2+' -gravity southwest -geometry '+EffLogo2s+'x'\
    + EffLogo2s+'+'+EffLogo2x+'+'+EffLogo2y+' -composite '\
    + '-pointsize '+EffPoints+' -gravity northwest '+multidraw\
    + ofile

    print("Generating: " + ofile)
    return(cmdstr)

os.chdir(curdir)
# *Image Magick*
os.system(magick_string(tmpdir,datdrp+'/EMCdata/logos',height,imgdir+'/'+ofile))
# IM will NOT! still be working when this python script returns (subprocess.call?)
if (view=='view'):
  cmd=viewer + ' ' + imgdir+'/'+ofile
  if opsys=='Windows':
    cmd=cmd.replace('/','\\')
  os.system(cmd)
