#!/usr/bin/env python3
# -*- coding: utf-8 -*-
# USAGE: HIMA8.py YYYYmmDDHHMM, where DD=01..31, HH=00..23, MM=00,10,20,30
#************************************************************************
#
# Japanese geostationary satellite Himawari-8 (HIMA8) at 144.7 degrees East
#
# See Hugo's EUMETCastView or in .../satpy/etc/readers/ahi_hrit.yaml
#------------------------------------------------------------------------
# EUMETSAT is EUMETCast disseminating 10 Minutes HIMA8 data in E1H-TPG-2
# Himawari-8 (HIMA8), at 144.7 noon is at about 03:00 UTC, instrument 'ahi'
#
# Channel | B01  | B02  | VIS  | B04  | B05  | B06  | IR4  | IR3  |
# Channel | B01  | B02  | B03  | B04  | B05  | B06  | B07  | B08  |
# WaveLen | 0.47 | 0.51 | 0.64 | 0.86 | 1.6  | 2.3  | 3.9  | 6.2  |
# --------+------+------+------+------+------+------+------+------+
# Channel | B09  | B10  | B11  | B12  | IR1  | B14  | IR2  | B16  |
# Channel | B09  | B10  | B11  | B12  | B13  | B14  | B15  | B16  |
# WaveLen | 6.9  | 7.3  | 8.6  | 9.6  | 10.4 | 11.2 | 12.4 | 13.3 |
#
# CH-3123 Belp     , 2021/01/09        License GPL3         (c) Ernst Lobsiger
#         Hilversum, 2022/01/27       License GPL3          (c) Rob Alblas
#
#************************************************************************

# Hugos's EUMETCastView, mpop, Himawari files use the upper Channel names
# satpy uses lower Channel names B01 .. B16 (loading 'VIS' does not work)
# But 'VIS' makes part of the file names that I am looking to decompress.
# Good natural FES images around 03:00 UTC, we decompress only files used


# I need
import os
import sys
import argparse
from os.path import expanduser

# Pytroll/SatPy needs these
from satpy import Scene
from glob import glob

# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

#Choose OS: uncomment one
opsys='Linux'
#opsys='Windows'

#Location where tools and data of satpy are located
home=expanduser("~")
toodrp = home+'/tools/satpy'
datdrp = home+'/tools/satpy'

def get_all_areas(sel=''):
  from satpy.resample import get_area_file
  areafiles=get_area_file()
  print('Areas:')
  for fs in areafiles:
    print('  '+fs+':')
    with open(fs) as f:
      for line in f:
        ln=line.strip('\n')
        if ln[len(ln)-1:] == ':' and ln[0]!=' ':
          ln=ln.strip(':')
          if sel=='' or sel in ln:
            print('    '+ln)

def is_segdir(rchan,Date):
  Yea=Date[:4]
  Mon=Date[4:6]
  Day=Date[6:8]
  return '/srv/rec_1/'+rchan+'/' + Yea + '/' + Mon + '/' + Day

# if segdir/<y>/<m>/<d> exists: return this, otherwise return segdir
def mkloc_ymd(segdir,y,m,d):
  osegdir=segdir+'/'+y+'/'+m+'/'+d
  if os.path.isdir(osegdir):
    return osegdir
  return segdir

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

if (opsys=='Windows'):
  convert_cmd=toodrp + '/EMCtools/exefiles/convert.exe'
  remove_cmd='del'
  bunzip2_cmd=toodrp + '/EMCtools/exefiles/7za x -so '
  bckclr='rgb(245,245,245)'
else:
  from shutil import which
  if which('magick'):
    convert_cmd='magick'
  elif which('convert'):
    convert_cmd='convert'
  else:
    sys.exit("ERROR: 'magick' or 'convert' not found.")

  remove_cmd='rm -f'
  bunzip2_cmd='bzip2 -dc'
  bckclr='rgb\(245,245,245\)'

# Needed for grid labels that have their own color setting (see above grid definition)
# This did segfault for a while and has then been fixed by Dave (may or may not work)!
# import aggdraw

# Why to hell is it not working?
# from satpy.utils import debug_on
# debug_on()

# Command line parsing
usage1 = 'Usage: ' + sys.argv[0] + ' -t YYYYmmDDHHMM [-src sourcedir] [-dst destinationdir] [-area <area>] [-composite <composite>]'

parser = argparse.ArgumentParser('HIMAx')
parser.add_argument('-t'        ,default='', help="time: YYYYmmDDHHMM")
parser.add_argument('-src'      ,default='', help="source dir., MUST be absolute path!")
parser.add_argument('-dst'      ,default='', help="destination dir.")
parser.add_argument('-o'        ,default='', help="output file")
parser.add_argument('-area'     ,default='', help="area, 'list' gives a list, 'list=<str>': items containing <str>")
parser.add_argument('-composite',default='', help="composite, 'list' gives a list")
parser.add_argument('-v'        ,default='noview',nargs='?',const='view', help="view")

args = parser.parse_args()
Dat=args.t
src=args.src
dst=args.dst
ofile=args.o
area=args.area
composite=args.composite
view=args.v
if opsys=='Windows':
  viewer='eog'
else:
  viewer='eog'

if (area[:4]=='list'):
  get_all_areas(area[5:])
  sys.exit()

if Dat=='':
  sys.exit(usage1)

satid='HIMA8'
rchan='E1H-TPG-2'

tsrvc='HVS-1'
rectype='SR-1'

# Get substrings
Yea=Dat[:4]
Mon=Dat[4:6]
Day=Dat[6:8]
Hou=Dat[8:10]
Min=Dat[10:]


# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

# Edit 1-3 parameter(s) below according to your file layout:
# My files are in a /MountPoint/Channel/YYYY/mm/DD structure
# No trailing / because this will make timestamps unreadable
# Windows toodrp = 'C:', datdrp = 'D:' / GNU/Linux use homedir
if (src==''):
  segdir=is_segdir(rchan,Dat)
else:
  segdir = src

if (segdir[len(segdir)-1]=='/'):
  segdir = segdir[:-1]

# try to search for YYYY/mm/DD location, extend segdir if exists 
#   (asume files are saved that way)
segdir=mkloc_ymd(segdir,Yea,Mon,Day)

print("Source dir.: " + segdir)

curdir=os.getcwd()

# Available composites in this script (far more in ../satpy/etc/composites/ahi.yaml)
# AVAILABLE: 'natural_color', 'true_color', 'overview', 'airmass', 'colorized_ir_clouds'
if (composite == ''):
  composite = 'true_color'

# Choose your area
if (area == ''):
  area = 'full_scan'
# area = 'australia'
# area = 'japan'
elif area == 'swath':
  area='full_scan'

print("Area     : " + area)
print("Composite: " + composite)

# Configure your individual overlays, either True/False or 1/0 do work
ADDgrid    =  True
ADDcoasts  =  True
ADDborders =  True
ADDrivers  =  True
ADDpoints  =  True

# OVRCache speeds up static overlays. These are stored in your .../EMCdata/cache
# With caching you will sometimes have to delete *.png OVRcache files manually !
OVRcache   =  False
# GEOcache allows for a resample cache of GEO satellites (speedy massproduction)
# These caches are stored in subdirectories with unique names .../nn_lut-*.zarr
GEOcache   =  False

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^


# Set temporary directory and image directory
tmpdir = datdrp + '/EMCdata/tmpdirs/xhima8'
if (dst==''):
  imgdir = datdrp + '/EMCdata/images/Himawari-8'
else:
  imgdir = dst
  if (imgdir[len(imgdir)-1]=='/'):
    imgdir = imgdir[:-1]

if (ofile==''):
  if (view=='view'):
    ofile='satpy_preview.jpg'
  else:
    ofile=satid+'-'+Dat[:8]+'-SLO-'+Dat[8:]+'-'+composite+'-'+area+'.jpg'

# Dir must exist
try:
   os.chdir(tmpdir)
except:
   sys.exit(tmpdir+': Cannot change to tmp directory ...')

# Clean tmp directory
os.system(remove_cmd+' IMG_DK01*')

# We only decompress what is actually needed, see file ahi.yaml
needed_files = {'natural_color'      : ['VIS','B04','B05'],
                'true_color'         : ['B01','B02','VIS','B04'],
                'overview'           : ['VIS','B04','IR1','B14'],
                'airmass'            : ['IR3','B10','B12','B14'],
                'colorized_ir_clouds': ['IR1'],
                'all'                : ['VI*','B*','IR*']}

# Some composite names are too long for IM annotation left of pic
abbreviation = {'natural_color'      : 'natural_color',
                'true_color'         : 'true_color',
                'overview'           : 'overview',
                'airmass'            : 'airmass',
                'colorized_ir_clouds': 'col_ir_clouds'}

#If composite isn't in list of needed files: just take all channels
if composite in needed_files:
  composite_neededfiles=composite
else:
  composite_neededfiles='all'

#If composite isn't in list of abbreviations: just take full name
if composite in abbreviation:
  composite_abbr=abbreviation[composite]
else:
  composite_abbr=composite

for b in needed_files[composite_neededfiles]:
    files = glob(segdir + '/' + 'IMG_DK01' + str(b) + '_' + Dat[:-1] + '*')
    # Decompress all n files found
    for f in files:
        ext=f[len(f)-4:len(f)]
        if (ext=='.bz2'):
          cmdstr=bunzip2_cmd+' ' + f + ' > ' + f[len(segdir)+1:-4]
          os.system(cmdstr)
        else:
          cmdstr='cp ' + f + ' ' + tmpdir
          os.system(cmdstr)
	

# Now all the files needed should be decompressed in tmpdir
files = glob(tmpdir + '/' + 'IMG_DK01*' + Dat[:-1] + '*')

# We should have got at least some good files
if files == []:
    sys.exit('Sorry, no good files found ...')

global_scene = Scene(filenames = files, reader = 'ahi_hrit')
if (composite=='list'):
  print('All composites:')
  composites=global_scene.available_composite_names()
  print(str(composites))
  quit()

global_scene.load([composite])

# This is a speedup by projection/area cache. It works for GEOstationary satellites only. 
if GEOcache:
    geocache = datdrp + '/EMCdata/cache/'+satid
else:
    geocache = None

# Full scan image has North on top, image size is 5500x5500 pixels
# Only 'airmass' has uniform resolution that can directly be saved
if area == 'full_scan':
    new_scene = global_scene.resample(resampler = 'native')
else:
    new_scene = global_scene.resample(area, radius_of_influence = 20000, cache_dir = geocache) #, reduce_data = False)

# Next is line by Christian Peters, thanks a lot!
# try:except: added after Graham Woolf's problems
try: # This is for real (multichannel) composites
    layers, height, width = new_scene[composite].shape
except: # In case of a single channel 'composite'
    height, width = new_scene[composite].shape

# Autoscale font_size (fs) of grid labels
fs = int((height + width) / 100)

# my_font = aggdraw.Font('red', datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
#           opacity = 255, size = fs) # aggdraw.Font takes only 4 arguments
# Font color can either be specified 'white' or (R, G, B) = (255, 255, 255)
# .. 'font': my_font} below may or may not work anymore now or in the future.
# If a direct font definition does not work, use the inline definition below:

# New interface to pycoast + POIs have been integrated 2020, thanks Martin !!
my_grid  = {'major_lonlat': (10,10), 'minor_lonlat': (2, 2),
            # Opacity 0 will hide the line, values 0 ... 255  EL
            'outline': (255, 255, 255) , 'outline_opacity': 255,
            'minor_outline': (200, 200, 200),'minor_outline_opacity': 127,
            'width': 1.5, 'minor_width': 1.0, 'minor_is_tick': False,
            'write_text': True, 'lat_placement': 'lr', 'lon_placement': 'b',
            'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf', 'font_size': fs}
            # minor_is_tick: False draws a line not ticks, True does ticks
            # label placement l,r,lr for latitude and t,b,tb for longitude

# Colors are specified as strings (140 HTML color names) or as triple (R, G, B) 8Bit.
# Best 'resolution' is automatically choosen according to area_def, no need to touch.
# Coasts also has keys ..,'fill': 'green', 'fill_opacity': 50} for land mass filling.
# 'level' defaults to 1 but should be choosen at least 2 for rivers (1 = 0 in GSHHG).
my_coasts  = {'outline': (255, 255,   0), 'width': 1.5, 'level': 1} # , 'resolution': 'i'}  # level 1 .. 4
my_borders = {'outline': (255,   0,   0), 'width': 1.0, 'level': 1} # , 'resolution': 'h'}  # level 1 .. 3
my_rivers  = {'outline': (  0,   0, 255), 'width': 1.0, 'level': 3} # , 'resolution': 'f'}  # level 1 .. 11

# List of POIs (Station, Cities, etc. ...)
my_poi_list = [((13.4052, 52.5214), 'Berlin'),
               (( 2.3522, 48.8566), 'Paris'),
               (( 0.1278, 51.5074), 'London'),
               ((-3.1880, 55.9539), 'Edinburgh'),
               ((-6.2598, 53.3514), 'Dublin'),
               (( 7.4989, 46.8918), 'Belp')]

# You may find other True Type Fonts (*.ttf) in /usr/share/fonts/truetype/...
# Windows 10 PRO users may find more True Type Fonts in C:\Windows\Fonts\ ...
my_points = {'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
             'font_size': 20, 'points_list': my_poi_list, 'symbol': 'circle',
             'ptsize': 20, 'outline': 'black', 'fill': 'red', 'width': 5.0,
             'outline_opacity': 255,'fill_opacity': 255, 'box_outline': 'navy',
             'box_linewidth': 3.0, 'box_fill': 'gold', 'box_opacity': 255}

# Cache is useful for speedup. Overlays are stored as tranparent *.png per satellite!
my_cache = {'file': datdrp + '/EMCdata/cache/' + satid + '/' + composite, 'regenerate': False}

# Setup indivdual overlays dictionary
my_overlays = {}
if ADDgrid:    # Add key grid
    my_overlays['grid'] = my_grid
if ADDcoasts:  # Add key coasts
    my_overlays['coasts'] = my_coasts
if ADDborders: # Add key borders
    my_overlays['borders'] = my_borders
if ADDrivers:  # Add key rivers
    my_overlays['rivers'] = my_rivers
if ADDpoints:  # Add key points
    my_overlays['points'] = my_points
if OVRcache:   # Add key cache
    my_overlays['cache'] = my_cache

# Enhance image with overlays and save *.png to temporary directory
new_scene.save_dataset(composite, satid+'.png',
          overlay = {'coast_dir': datdrp + '/EMCdata/gshhg-shp',
         'overlays' : my_overlays}) # 2^6=32 combinations should do

# ****************************************************************
# NEW POST PROCESSING: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# ****************************************************************

# ImageMagick (IM) postprocessing, added directory logos in EMCdata
# logo1 = datdrp + '/EMCdata/logos/GNU-Linux-Logo-Penguin-SVG.png'
#logo1 = datdrp + '/EMCdata/logos/JMA_947x969.gif'
#logo2 = datdrp + '/EMCdata/logos/PyTROLL_400x400.jpg'

# Use IM to annotate (maybe also -equalize) and convert to *.jpg
# www.imagemagick.org/Usage/crop/#splice or .../annotating/#labeling
# This is the syntax used -draw "text OffsetX,OffsetY 'Hello World!'"
# This annotates to the left, numbers depend on original image size
# Colors can be {dark,light}blue, etc. or rgb(R,G,B) with \( escaped

def magick_string(tmpdir,loc_logo,EffHeight,ofile):
    """
    It seems ImageMagick under Windows cannot cope with multiline text
    that includes \n newlines. We have to draw every line separately.
    This scales all pixel values for IM according to the image height.
    Should make it easier to use annotation and logos at the left side.
    This has been optimized using MSG4 full scan with 3712x3712 pixels.
    """

    logo1 = loc_logo + '/'+'JMA_947x969.gif'
    logo2 = loc_logo + '/PyTROLL_400x400.jpg'

    OptHeight = 3712
    OptSplice = 640
    OptLogo1s = 540
    OptLogo1x = 50   # 40 for 'GNU-Linux-Logo-Penguin-SVG.png'
    OptLogo1y = 40   # 60 for 'GNU-Linux-Logo-Penguin-SVG.png'
    OptLogo2s = 530
    OptLogo2x = 55
    OptLogo2y = 55
    OptPoints = 90

    fac = EffHeight/OptHeight/2
    EffSplice = str(int(fac*OptSplice+0.5))
    EffLogo1s = str(int(fac*OptLogo1s+0.5))
    EffLogo1x = str(int(fac*OptLogo1x+0.5))
    EffLogo1y = str(int(fac*OptLogo1y+0.5))
    EffLogo2s = str(int(fac*OptLogo2s+0.5))
    EffLogo2x = str(int(fac*OptLogo2x+0.5))
    EffLogo2y = str(int(fac*OptLogo2y+0.5))
    EffPoints = str(int(fac*OptPoints+0.5))

    satname=satid
    sensor= 'ahi'

    legend = [' \'DATE:\'\" '     , ' \''+Yea+'/'+Mon+'/'+Day+'\'\" ',
              ' \'TIME:\'\" '     , ' \''+Hou+':'+Min+' UTC'+'\'\" ',
              ' \'SOURCE:\'\" '   , ' \'EUMETCast\'\" ',
              ' \'SERVICE:\'\" '  , ' \''+tsrvc+'\'\" ',
              ' \'CHANNEL:\'\" '  , ' \''+rchan+'\'\" ',
              ' \'RECEIVER:\'\" ' , ' \''+rectype+'\'\" ',
              ' \'SATELLITE:\'\" ', ' \''+satname+'\'\" ',
              ' \'SENSOR:\'\" '   , ' \''+sensor+'\'\" ',
              ' \'COMPOSITE:\'\" ', ' \''+composite_abbr+'\'\" ',
              ' \'AREA:\'\" '     , ' \''+area+'\'\" ']
    multidraw = ''
    yoff = fac*470
    for n in range(0, len(legend)):
        if (n % 2) == 0:
            xoff = fac*40
            yoff += fac*150
        else:
            xoff = fac*60
            yoff += fac*100
        multidraw += '-draw \"text +'+str(int(xoff+0.5))+'+'+str(int(yoff+0.5))+legend[n]

    ifile=tmpdir+'/'+satid+'.png'
    cmdstr = convert_cmd+' '+ifile+' '\
    + '-font Nimbus-Sans '\
    + '-gravity west -background '+ bckclr +' '\
    + '-splice '+EffSplice+'x0 '\
    + logo1+' -gravity northwest -geometry '+EffLogo1s+'x'\
    + EffLogo1s+'+'+EffLogo1x+'+'+EffLogo1y+' -composite '\
    + logo2+' -gravity southwest -geometry '+EffLogo2s+'x'\
    + EffLogo2s+'+'+EffLogo2x+'+'+EffLogo2y+' -composite '\
    + '-pointsize '+EffPoints+' -gravity northwest '+multidraw\
    + ofile

    print("Generating: " + ofile)
    return(cmdstr)

os.chdir(curdir)
# *Image Magick*
os.system(magick_string(tmpdir,datdrp+'/EMCdata/logos',height,imgdir+'/'+ofile))
# IM will NOT! still be working when this python script returns (subprocess.call?)
if (view=='view'):
  cmd=viewer + ' ' + imgdir+'/'+ofile
  if opsys=='Windows':
    cmd=cmd.replace('/','\\')
  os.system(cmd)
