#************************************************************************
#
# This module provides extra GEO functions for generating *GEO* satellite
# products using GEO funcs like MSG.py etc. containing extended options, and
# with 'Satpy Processing System' (SPS) advanced Kit Version 4.1.
# One copy of the file must reside in the same direcory with GEO scripts.
#
#    Hilversum, 2024/04/01        License GPL3        (c) Rob Alblas
#
#************************************************************************
import argparse
from argparse import RawTextHelpFormatter
import platform
import os,sys

from GEOstuff import get_prodir_areal_areaf
from satpy.resample import get_area_def

#Define viewer. If '': use internal viewer.
Viewer=''

OS = platform.system()

class options:
  def __init__(self):
    self.Date=None
    self.sat=None
    self.src=None
    self.bulk=None
    self.dst=None
    self.ofile=None
    self.area=None
    self.composites=None
    self.for_movie=None
    self.do_view=None
    self.ovl=None

    self.coasts  = False
    self.borders = False
    self.rivers  = False
    self.lakes   = False
    self.grid    = False
    self.cities  = False
    self.points  = False
    self.station = False
    self.legend  = False
    self.default = False



def resize_image(area,maxsize=1080):
  sarea=get_area_def(area)
  if sarea.width > maxsize:
    nwidth=maxsize
    nheight=int(sarea.height*maxsize/sarea.width)
    new_id=sarea.area_id+'_r'
    print('Resizing from '+str(sarea.width)+'x'+str(sarea.height)+' to '+str(nwidth)+'x'+str(nheight)+', new id='+new_id)
    sarea=get_area_def(area).copy(width=nwidth,height=nheight,area_id=new_id)

  area=sarea.area_id
  return area,sarea

def get_args_geo(prog):
  parser = argparse.ArgumentParser(formatter_class=RawTextHelpFormatter)
#  parser = argparse.ArgumentParser(prog)
  parser.add_argument('-t'        ,default=''      , help="time: YYYYmmDDHHMM")
  if prog=='MTI':
    parser.add_argument('-sat'      ,default='1'     , help="satnr: 1")
  elif prog=='MSG':
    parser.add_argument('-sat'      ,default='4'     , help="satnr: 1...4, 1n...4n")
  else:
    parser.add_argument('-sat'      ,default=''      , help="satnr: 8, 9, 16, ...")

  parser.add_argument('-src'      ,default=''      , help="source dir., MUST be absolute path!")
  parser.add_argument('-b'        ,default=''      , help="bulk: y |n.")
  parser.add_argument('-dst'      ,default=''      , help="destination dir.")
  parser.add_argument('-o'        ,default=''      , help="output file")
  parser.add_argument('-area'     ,default=''      , help="area") #, 'list' gives a list, 'list=<str>': items containing <str>")
  parser.add_argument('-composite',default=''      , help="composite") #, 'list' gives a list")
  parser.add_argument('-m'        ,default=False,nargs='?',const=True, help="reduced size, for movie")
  parser.add_argument('-ovl'      ,default='D'     , help="overlays: c=coast,  b=borders, g=grid, \n"+\
                                                          "          r=rivers, l=lakes,   C=cities,\n"+\
                                                          "          p=points, s=station, L=Legend\n"+\
                                                          "          A=all, N=none, D or no -ovl: defaults")
  parser.add_argument('-v'        ,default=False,nargs='?',const=True, help="Start viewer")

  args     = parser.parse_args()

  opts=options()
  opts.Date     = args.t
  opts.sat      = args.sat
  opts.src      = args.src

  if args.b.lower()=='y':
    opts.bulk=True
  elif  args.b.lower()=='n':
    opts.bulk=False
    
  opts.dst       = args.dst
  opts.ofile     = args.o
  opts.area      = args.area
  opts.composites= args.composite
  opts.for_movie = args.m
  opts.do_view   = args.v

  opts.ovl        = args.ovl
  if "c" in opts.ovl or "A" in opts.ovl:
    opts.coasts  = True
  if "b" in opts.ovl or "A" in opts.ovl:
    opts.borders = True
  if "r" in opts.ovl or "A" in opts.ovl:
    opts.rivers = True
  if "l" in opts.ovl or "A" in opts.ovl:
    opts.lakes = True
  if "g" in opts.ovl or "A" in opts.ovl:
    opts.grid = True
  if "C" in opts.ovl or "A" in opts.ovl:
    opts.cities = True
  if "p" in opts.ovl or "A" in opts.ovl:
    opts.points = True
  if "x" in opts.ovl or "A" in opts.ovl:
    opts.station = True
  if "L" in opts.ovl or "A" in opts.ovl:
    opts.legend = True
  if "D" in opts.ovl:
    opts.default = True

  if opts.Date=='':
    sys.exit('Error: Option -t needed.')
  if opts.sat=='':
    sys.exit('Error: Option -sat needed.')

  print_ovl(opts)

  return opts
#  if bulk or nobulk:
#    return asat,Dat,[src,bulk],dst,ofile,area,composite,for_movie,do_view
#  else:
#    return asat,Dat,src,dst,ofile,area,composite,for_movie,do_view

def print_ovl(opts):
  print("Add overlays:")
  if opts.coasts:
    print("  coast")
  if opts.borders:
    print("  boarders")
  if opts.rivers:
    print("  rivers")
  if opts.lakes:
    print("  lakes")
  if opts.grid:
    print("  grid")
  if opts.cities:
    print("  cities")
  if opts.points:
    print("  points")
  if opts.station:
    print("  station")
  if opts.legend:
    print("  legend")
  if opts.default:
    print("  Default overlays")

def split_Dat(Dat):
  # Get substrings
  Yea=Dat[:4]
  Mon=Dat[4:6]
  Day=Dat[6:8]
  Hou=Dat[8:10]
  Min=Dat[10:]

  return Yea, Mon, Day, Hou, Min

# input contains: <magick_command>  
def split_ofile(magick):
  splstr=magick.rsplit(' ',1)
  prog=splstr[0]
  splstr=splstr[1].rsplit('/',1)
  loc=splstr[0]
  fn=splstr[1]
  return prog,loc,fn

# recreate magick command as returned by get_magick()
# to change location and filename
def recreate_cmd(magick,dst,ofile,curdir):
  prog,loc,fn=split_ofile(magick)

  if dst!='':
    if OS == 'Linux':
      if dst[0]=='/' or dst[0]=='~':
        loc=dst
      else:
        loc=curdir+'/'+dst
    elif OS == 'Windows':      # only abs. path possible, yet
      loc=dst

  if ofile!='':
    fn=ofile

  if dst!='' or ofile!='':
    magick=prog+' '+loc+'/'+fn
  return magick,loc+'/'+fn

def test_file_exist(sat,dst,subdir,ofile):
  if dst=='':
    nn=get_prodir_areal_areaf(sat,'','')
    dst=nn[0]+'/'+subdir
  pfn=dst+'/'+ofile
  print('output='+pfn)
  if os.path.isfile(pfn):
    return pfn
  return ''

# A simple viewer
from tkinter import *
from PIL import ImageTk, Image
def satpy_viewer(pfn):
  if Viewer=='':
    maxwidth=1000
    maxheight=1000
    root = Tk()
    root.title(pfn)
    img1 = Image.open(pfn)
    w1=img1.width
    h1=img1.height
    sfactx=maxwidth/w1
    sfacty=maxheight/h1
    sfact=min(sfactx,sfacty)
    w1=int(w1*sfact)
    h1=int(h1*sfact)
    img2=img1.resize((w1,h1))

    img = ImageTk.PhotoImage(img2)
    label = Label(image=img)
    label.grid()
    root.mainloop()
  else:
    os.system(Viewer+' '+pfn+'&')

