############################################################################################
# Shared functions for Satpy, all LEO satellites
#------------------------------------------------------------------------
#
# Hilversum   , 2022/04/09       License GPL3          (c) Rob Alblas (Ernst Lobsiger)
#
#************************************************************************
############################################################################################

import math
import sys
import os
import re
from glob import glob
from datetime import datetime, timedelta
from pyorbital.orbital import Orbital
from satpy import Scene, MultiScene

from satpy_settings import opsys,toodrp,Viewer


class pos:
  def __init__(self):
    lon=0.
    lat=0.
    ran=0.
    cosmax=0.

# Satellite/Eumetcast specific items
class sat:
#  def __init__(self):
    bzipped=False
    tarred=False
    date=''
    d1=0
    d2=0
    lenseg=0
    paired_files=False
    files=[]                  # 3 or 4-dim array with date, time, filename and (fy3d) second filename
    cfiles=[]                 # 3 or 4-dim array with date, time, filename and (fy3d) second filename; current day
    pfiles=[]                 # 3 or 4-dim array with date, time, filename and (fy3d) second filename; prev. day
    nfiles=[]                 # 3 or 4-dim array with date, time, filename and (fy3d) second filename; next day
    member_files=[]
    orb=None
    pos
    cosmax=0.
    hourmax=0
    timmax=0
    reader=''
    segdir=''
    rchan=''
    tsrvc=''
    sensor=''
    satname=''
    satname_st=''            # satname for spacetrack
    pre_filename=''
    logo=''

# Generate parameters etc.
class gen:
    opsys            = ''
    composite        = ''
    area             = ''
    area_noselect    =False
    daynight_noselect=False
    day_pass         = True
    ADDgrid          = True
    ADDcoasts        = True
    ADDborders       = True
    ADDrivers        = True
    ADDpoints        = True
    OVRcache         = False
    remove_cmd       = ''
    rmdir_cmd        = ''
    convert_cmd      = ''
    bunzip2_cmd      = ''
    unzip_cmd =      ''
    tar_cmd          = ''
    receiver         = 'SR-1'
    viewer           = ''
    dry=0

class mestype:
  quiet=0
  info=1
  warn=2
  error=3


def print_dbg(message,n=mestype.quiet):
  meslvl=mestype.quiet
  if n>mestype.quiet and n>=meslvl:
    print(message)

# if segdir/<y>/<m>/<d> exists: return this, otherwise return segdir
def mkloc_ymd(segdir,y,m,d):
  osegdir=segdir+'/'+y+'/'+m+'/'+d
  if os.path.isdir(osegdir):
    return osegdir
  return segdir

# init. gen-class
def Set_Gen(day_pass,composite,area,dry,viewer):
  igen=gen
  igen.opsys=opsys
  igen.day_pass=day_pass
  igen.composite=composite
  igen.area=area
  if area=='swath':
    igen.area_noselect     =True
    igen.daynight_noselect =True

  if ((viewer) and (str(viewer)!='1')):
    igen.viewer=viewer
  else:
    igen.viewer=Viewer
  igen.dry=dry
  if (opsys=='Windows'):
    igen.convert_cmd=toodrp + '/EMCtools/exefiles/convert.exe'

    igen.remove_cmd='del'
    igen.rmdir_cmd='rmdir /s /q' # is this OK with wildcards?
    igen.bunzip2_cmd=toodrp + '/EMCtools/exefiles/7za x -so'
    igen.tar_cmd=toodrp + '/EMCtools/exefiles/7za x'
  else:
    from shutil import which
    if which('magick'):
      igen.convert_cmd='magick'
    elif which('convert'):
      igen.convert_cmd='convert'
    else:
      sys.exit("ERROR: 'magick' or 'convert' not found.")

    igen.remove_cmd='rm -f'
    igen.rmdir_cmd='rm -rf'
    igen.bunzip2_cmd='bzip2 -dc'
    igen.unzip_cmd='unzip -o'
    igen.tar_cmd='tar -xf'
  return igen

def Set_Overlay(gen,ADDgrid,ADDcoasts,ADDborders,ADDrivers,ADDpoints):
  gen.ADDgrid    = ADDgrid
  gen.ADDcoasts  = ADDcoasts
  gen.ADDborders = ADDborders
  gen.ADDrivers  = ADDrivers
  gen.ADDpoints  = ADDpoints

# Collect files; do_extend=True: also some yesterday's and tomorrow's files
# Format expected:
#   sfrmt+YYYYmmdd+tfrmt+HHMMSS+efrmt
#   sfrmt+YYYYDOY+tfrmt+HHMMSS+efrmt
def collect_files(sfrmt,Date,tfrmt,efrmt,is_doy,do_extend):
  Yea = Date[:4]
  Mon = Date[4:6]
  Day = Date[6:]
  pfiles=[]
  nfiles=[]
  files=[]

  # Files of current day
  if is_doy:
    dt = datetime(int(Yea), int(Mon), int(Day))
    YYYYDOY = dt.strftime('%Y%j')
    print_dbg('Filter: ' + sfrmt + YYYYDOY + efrmt,mestype.info)
    files = glob(sfrmt + YYYYDOY + efrmt)
  else:
    print_dbg('Filter: ' + sfrmt + Date + efrmt,mestype.info)
    files = glob(sfrmt + Date + efrmt)

  if do_extend:
    print_dbg('Add some prev/next day files',mestype.info)
    if is_doy:
      dt = datetime(int(Yea), int(Mon), int(Day))
      YYYYDOY = dt.strftime('%Y%j')
      pDate=int(YYYYDOY)-1
      nDate=int(YYYYDOY)+1
    else:
      dt1 = datetime(int(Yea), int(Mon), int(Day))
      dt2 = timedelta(days=1)

      # previous day, from 20:00 on
      dt=dt1-dt2
      Day=str(dt.strftime('%d'))
      Mon=str(dt.strftime('%m'))
      Yea=str(dt.strftime('%Y'))
      pDate=Yea + Mon + Day

      # next day, from 20:00 on
      dt=dt1+dt2
      Day=str(dt.strftime('%d'))
      Mon=str(dt.strftime('%m'))
      Yea=str(dt.strftime('%Y'))
      nDate=Yea + Mon + Day

    print_dbg('Filter yesterday: ' + sfrmt + str(pDate) + tfrmt + '22'+efrmt,mestype.info)
    print_dbg('Filter yesterday: ' + sfrmt + str(pDate) + tfrmt + '23'+efrmt,mestype.info)
    pfiles = glob(sfrmt + str(pDate) + tfrmt + '22'+efrmt)
    pfiles = pfiles+glob(sfrmt + str(pDate) +tfrmt + '23'+efrmt)

    print_dbg('Filter tomorrow : ' + sfrmt + str(nDate) + tfrmt + '00'+efrmt,mestype.info)
    print_dbg('Filter tomorrow : ' + sfrmt + str(nDate) + tfrmt + '01'+efrmt,mestype.info)
    nfiles = glob(sfrmt + str(nDate) + tfrmt + '00'+efrmt)
    nfiles = nfiles+glob(sfrmt + str(nDate) + tfrmt + '01'+efrmt)

  print_dbg('Nr. today\'s     files: ' + str(len(files)),mestype.info)
  if do_extend:
    print_dbg('Nr. yesterday\'s files: ' + str(len(pfiles)),mestype.info)
    print_dbg('Nr. tomorrow\'s  files: ' + str(len(nfiles)),mestype.info)

  return files,pfiles,nfiles

# Read filenames from file
def get_files(flst,trail=''):
  flist=[]
  with open(flst) as f:
    for line in f:
      fn=line.strip('\n')
      if (fn[len(fn)-len(trail):] == trail):
        flist.append(fn)
    f.close()
  return (flist,[],[])

# Select potential files based on area and time-of-day (day, night)
def use_this_pass(day_pass,d1,d2,Hou):
  if d2 > d1:
    if (day_pass):
      if (int(Hou) > d1 and int(Hou) < d2):
        return True; 
    else:
      if (int(Hou) < d1 or int(Hou) > d2):
        return True;
  else:
    if (day_pass):
      if (int(Hou) > d1 or int(Hou) < d2):
        return True; 
    else:
      if (int(Hou) < d1 and int(Hou) > d2):
        return True;

  return False

# Select files:
#  ( lat, lon) = center area
#  (slat,slon) = pos. sat at center of segment
#    time center of segment: in filename, see load_filetbl()
def select_files(satpar,genpar):
  d1=satpar.d1
  d2=satpar.d2
  lenseg=satpar.lenseg
  daynight_noselect=False
  area_noselect=False

  if genpar!=None:
    daynight_noselect=genpar.daynight_noselect
    area_noselect    =genpar.area_noselect

  day_pass=genpar.day_pass

  # center area
  lon=satpar.pos.lon
  lat=satpar.pos.lat
  ran=satpar.pos.ran

  dr = math.pi/180.0
  sinlat = math.sin(lat*dr)
  coslat = math.cos(lat*dr)
  cosran = math.cos(ran*dr)

  goodfiles=[]
  goodtimes=[]
  if (satpar.cosmax==0):
    satpar.hourmax=-1
  # Files are all files of a certain day (cfiles) and some prev+next day files (pfiles/nfiles); see 'glob' in main script
  for filess in satpar.cfiles,satpar.pfiles,satpar.nfiles:
    for fdt in filess:
      fname=fdt[2]
      Yea = fdt[0][0:4]
      Mon = fdt[0][4:6]
      Day = fdt[0][6:8]
      Hou = fdt[1][0:2]
      Min = fdt[1][2:4]
      Sec = 0                      #sec not always available in fdt!
      if ((filess==satpar.cfiles) or ((int(satpar.hourmax)<12 and int(Hou)>20) or (int(satpar.hourmax)>12 and int(Hou)<3))):
        # Day crossings between d1 and d2
        # Night crossings outside d1 to d2
        # no-select: take all
        if daynight_noselect or use_this_pass(day_pass,d1,d2,Hou):
          # Segments are 85 - 86 seconds long, let's advance to the center
          dt = datetime(int(Yea), int(Mon), int(Day), int(Hou), int(Min),
                        int(Sec)) + timedelta(seconds = lenseg/2)
          # Get longitide, latitude, altitude of sat at center of segment
          (slon, slat, salt) = satpar.orb.get_lonlatalt(dt)

          # Use Nautical Triangle for arc distance
          cosarc = sinlat * math.sin(slat*dr) + \
                   coslat * math.cos(slat*dr) * math.cos((lon-slon)*dr)
          # Within search range, maybe more than 1 orbit
          # some sats have info in 2 files per segment
          if (satpar.paired_files):
            gname=fdt[3]
            if cosarc > cosran or area_noselect:
              goodfiles.append(fname)
              goodtimes.append(int(dt.timestamp()))
              goodfiles.append(gname)
              goodtimes.append(int(dt.timestamp()))
              # We only want the best orbit
              if cosarc > satpar.cosmax:
                satpar.cosmax = cosarc
                satpar.timmax = int(dt.timestamp())
          else:
            if cosarc > cosran or area_noselect:
              #print('cosarc='+str(math.acos(cosarc)*180./3.14159)+' cosran='+str(math.acos(cosran)*180./3.14159) )
              goodfiles.append(fname)
              goodtimes.append(int(dt.timestamp()))
              # We only want the best orbit (only use files from wanted day)
              if filess==satpar.cfiles:
                if cosarc > satpar.cosmax:
                  satpar.cosmax = cosarc
                  satpar.timmax = int(dt.timestamp())
                  satpar.hourmax=Hou

  nr_goodfiles=0
  for gf in goodfiles:
    nr_goodfiles=nr_goodfiles+1

  for gf in goodfiles:
    print_dbg('  good files: --> ' + str(gf),mestype.info)
    break
  if nr_goodfiles > 1:
    print_dbg('  And '+str(nr_goodfiles-1)+' more good files',mestype.info)

  return(goodfiles,goodtimes)

# Stacked: Select files east or west of center pass
def sel_pass_side(goodfiles,goodtimes,satp,genp,do_east,scenes):
  numpass=0
  if (do_east):
    ta = min(goodtimes)
    tb=satp.timmax
  else:
    tb = max(goodtimes)
    ta=satp.timmax

  while tb - ta > 1500:
    passfiles = []
    n = len(goodtimes) - 1
    while n >= 0:
      if (((do_east) and (goodtimes[n] - ta < 1500)) or
          ((not do_east) and (tb-goodtimes[n] < 1500))):
        passfiles.append(goodfiles[n])
        goodfiles.pop(n)
        goodtimes.pop(n)
      n -= 1

    if goodtimes == []:
       break
    if (do_east):
      ta = min(goodtimes)
    else:
      tb = max(goodtimes)

    numpass += 1
    passfiles = sorted(passfiles)
    if (do_east):
      print_dbg('East Pass '+ str(numpass),mestype.info)
    else:
      print_dbg('West Pass '+ str(numpass),mestype.info)
    for n in range(0, len(passfiles)):
        print ('%2d --> ' % (n+1), passfiles[n])

    if not genp.dry:
      if (satp.bzipped):
        do_bunzip(passfiles,len(satp.segdir)+1)

      if (satp.tarred):
        passfiles=do_untar(passfiles,len(satp.segdir)+1,satp.member_files)

      if (len(passfiles) > 0):
        scenes.append(Scene(filenames = passfiles, reader = satp.reader))

  return(goodfiles,goodtimes,scenes)

# Stacked: Select remaining files (center pass)
def sel_pass_rest(restfiles,satp,genp,scenes):
  numpass=1
  print_dbg('Center Pass '+ str(numpass),mestype.info)
  for n in range(0, len(restfiles)):
    print ('%2d --> ' % (n+1), restfiles[n])

  if not genp.dry:
    if (satp.bzipped):
      do_bunzip(restfiles,len(satp.segdir)+1)

    if (satp.tarred):
      restfiles=do_untar(restfiles,len(satp.segdir)+1,satp.member_files)

    if restfiles != []:
      scenes.append(Scene(filenames = restfiles, reader = satp.reader))
  return(scenes)

# Not stacked: select single, best (center) pass
def sel_pass_center(goodfiles,goodtimes,satp,genp):
  scn=[]
  bestfiles=[]
  # Only take segment files of the best orbit
  for n in range(0, len(goodfiles)):
      if genp.area=='swath' or math.fabs(goodtimes[n] - satp.timmax) < 3000:
          bestfiles.append(goodfiles[n])

  # Sorting seems not necessary?!
  bestfiles = sorted(bestfiles)

  # Now use bestfiles for image
  if bestfiles == []:
      sys.exit('Sorry, no good files found ...')

  print_dbg("Found " + str(len(bestfiles)) + " potential files",mestype.info)
  # Uncomment 4-5 lines for good old print debug
  for n in range(0, len(bestfiles)):
     if n==0 or n==len(bestfiles)-1:
       print_dbg(f'{n+1:2d} --> {bestfiles[n]:s}',mestype.info)
     if n==1:
       print_dbg('....',mestype.info)

  if not genp.dry:
    if (satp.bzipped):
      do_bunzip(bestfiles,len(satp.segdir)+1)

    if (satp.tarred):
      bestfiles=do_untar(bestfiles,len(satp.segdir)+1,satp.member_files)

    # Swath image size is 3200 X (768) * NumberOfSegments (MX-Bands)
    if satp.reader=='':
      scn = Scene(filenames = bestfiles)
    else:
      scn = Scene(filenames = bestfiles, reader = satp.reader)
  return scn


# Bunzip files in 'files_to_unzip', returns name unzipped file in this list
def do_bunzip(files_to_unzip,offset):
  # Decompress all n passfiles found, if needed
  for n in range (0, len(files_to_unzip)):
    filename = files_to_unzip[n][offset:-4]
    ext=files_to_unzip[n][len(files_to_unzip[n])-4:len(files_to_unzip[n])]
    if (ext=='.bz2'):
      filename = files_to_unzip[n][offset:-4]
      cmdstr = gen.bunzip2_cmd+' ' + files_to_unzip[n] + ' > ' + filename
      files_to_unzip[n] = filename
      os.system(cmdstr)
    if (ext=='.zip'):
      filename = files_to_unzip[n][offset:-4] + '.nat'
      cmdstr = gen.unzip_cmd+' ' + files_to_unzip[n]
      files_to_unzip[n] = filename
      os.system(cmdstr)

def do_untar(files_to_untar,offset,member_files):
  untarred_files=[]
  for n in range (0, len(files_to_untar)):
    dname = files_to_untar[n][offset:-4]
    cmdstr = gen.tar_cmd+' ' + files_to_untar[n]
    for m in member_files:
      cmdstr += ' ' + dname + '/' + m
    os.system(cmdstr)
    untarred_files = untarred_files + glob(dname+'/*')
  return untarred_files

# Set lon/lat/ran for selected area
# See also: EMCtools/pppconfig/areas.yaml
# (or: miniconda3/envs/pytroll/lib/python3.10/site-packages/satpy/etc/areas.yaml
# Note: ran needs to be calculated some way, depends on size area. 
#   Too small: needed files not processed
#   Too big: unneeded files processed
def set_lonlat(area):
  from satpy_areas import get_area_info
  a=get_area_info(area)
  if a != None:
    pos.lat=a.lat
    pos.lon=a.lon
    pos.ran=a.ran
    pos.cosmax=0
    return pos
  else:
    pos.lat=0
    pos.lon=0
    pos.ran=90
    pos.cosmax=0
    return pos
  return None

#list all defined areas specified in areas.yaml
def list_all_areas(sel=''):
  from satpy.resample import get_area_file
  areafiles=get_area_file()
  print('Areas:')
  for fs in areafiles:
    print('  '+fs+':')
    with open(fs) as f:
      for line in f:
        ln=line.strip('\n')
        if ln[len(ln)-1:] == ':' and ln[0]!=' ':
          ln=ln.strip(':')
          if sel=='' or sel in ln:
            print('    '+ln)

# Remove substring 'substr' from s
# E.g. s='abcdef', substr='de' ==> 'abcf'
def rm_substr(s, substr):
    # type: (str, str) -> str
    return re.subn(substr, '', s)[0]

#Define output file if ofile is ''
def ofile_name(satpar,genpar,ofile,Dat):
  if (ofile==''):
    dt = datetime.fromtimestamp(satpar.timmax)
    tfname = dt.strftime('%H%M')
    if (genpar.day_pass):
      ofile=satpar.satname+'-'+Dat+'-DAY-'+tfname+'-'+genpar.composite+'-'+genpar.area+'.jpg'
    else:
      ofile=satpar.satname+'-'+Dat+'-NIG-'+tfname+'-'+genpar.composite+'-'+genpar.area+'.jpg'

  return ofile

# Create scene or multiscene and create base picture
def create_scene(satpar,genpar,gf,gt,composite,do_stack,datdrp):
  if (do_stack and composite!='list' and genpar.area!='swath'):
    scenes=[]
    (gf,gt,scenes)=sel_pass_side(gf,gt,satpar,genpar,True,scenes)
    (gf,gt,scenes)=sel_pass_side(gf,gt,satpar,genpar,False,scenes)

    scn=sel_pass_rest(gf,satpar,genpar,scenes)

    if genpar.dry:
      sys.exit('Stop, dry mode')

    mscn = MultiScene(scn)
    mscn.load([genpar.composite])

    new_mscn = mscn.resample(genpar.area, radius_of_influence = 5000) #, reduce_data = False)
    new_scn = new_mscn.blend()

    # For some reason, in case of hdf5 'save_dataset' in create_pic()
    #   MUST be in same func!
    # h5 error solved with:
    #   mscn.resample() 
    #   mscn.blend() 
    # instead of 
    #   new_mscn = mscn.resample() 
    #   new_scn = new_mscn.blend() 
    # but gives another error:
    #   blend "arguments without labels along dimension 'y' cannot be aligned because they have different dimension sizes: {3824, 7664}"

  else:
    scn=sel_pass_center(gf,gt,satpar,genpar)

    if genpar.dry:
      sys.exit('Stop, dry mode')

    if (composite=='list'):
      print('All composites:')
      composites=scn.available_composite_names()
      print(str(composites))
      sys.exit('Stop')

    scn.load([genpar.composite])
    if genpar.area=='swath':
      new_scn=scn
    else:
      new_scn=scn.resample(genpar.area, radius_of_influence = 4000)

  # Next muts be in same function, for some reason
  create_pic(satpar,genpar,new_scn,datdrp)
  return new_scn

# Create picture including overlays, but without legend
def create_pic(satpar,genpar,new_scn,datdrp):
  composite=genpar.composite
  area=genpar.area
  ADDgrid    =  genpar.ADDgrid
  ADDcoasts  =  genpar.ADDcoasts
  ADDborders =  genpar.ADDborders
  ADDrivers  =  genpar.ADDrivers
  ADDpoints  =  genpar.ADDpoints

# OVRCache speeds up static overlays. These are stored in your .../EMCdata/cache
# Never use this with homebrew or SatPy's dynamic areas like omec_bb and laea_bb
# With caching you will sometimes have to delete *.png OVRcache files manually !
  OVRcache   =  genpar.OVRcache

  # Next is line by Christian Peters, thanks a lot!
  # try:except: added after Graham Woolf's problems
  try: # This is for real (multichannel) composites
      layers, height, width = new_scn[composite].shape
  except: # In case of a single channel 'composite'
      height, width = new_scn[composite].shape

  # Autoscale font_size (fs) of grid labels
  fs = int((height + width) / 100)

  # my_font = aggdraw.Font('red', datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
  #           opacity = 255, size = fs) # aggdraw.Font takes only 4 arguments
  # Font color can either be specified 'white' or (R, G, B) = (255, 255, 255)
  # .. 'font': my_font} below may or may not work anymore now or in the future.
  # If a direct font definition does not work, use the inline definition below:

  # New interface to pycoast + POIs have been integrated 2020, thanks Martin !!
  my_grid  = {'major_lonlat': (10,10), 'minor_lonlat': (2, 2),
              # Opacity 0 will hide the line, values 0 ... 255  EL
              'outline': (255, 255, 255) , 'outline_opacity': 255,
              'minor_outline': (200, 200, 200),'minor_outline_opacity': 127,
              'width': 1.5, 'minor_width': 1.0, 'minor_is_tick': False,
              'write_text': True, 'lat_placement': 'lr', 'lon_placement': 'b',
              'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf', 'font_size': fs}
              # minor_is_tick: False draws a line not ticks, True does ticks
              # label placement l,r,lr for latitude and t,b,tb for longitude

  # Colors are specified as strings (140 HTML color names) or as triple (R, G, B) 8Bit.
  # Best 'resolution' is automatically choosen according to area_def, no need to touch.
  # Coast also has keys ... 'fill': 'green', 'fill_opacity': 50} for land mass filling.
  # 'level' defaults to 1 but should be choosen at least 2 for rivers (1 = 0 in GSHHG).
  my_coasts  = {'outline': (255, 255,   0), 'width': 1.5, 'level': 1} # , 'resolution': 'i'}  # level 1 .. 4
  my_borders = {'outline': (255,   0,   0), 'width': 1.0, 'level': 1} # , 'resolution': 'h'}  # level 1 .. 3
  my_rivers  = {'outline': (  0,   0, 255), 'width': 1.0, 'level': 3} # , 'resolution': 'f'}  # level 1 .. 11

  # List of POIs (Station, Cities, etc. ...)
  my_poi_list = [((13.4052, 52.5214), 'Berlin'),
                 (( 2.3522, 48.8566), 'Paris'),
                 (( 0.1278, 51.5074), 'London'),
                 ((-3.1880, 55.9539), 'Edinbourgh'),
                 ((-6.2598, 53.3514), 'Dublin'),
                 (( 7.4989, 46.8918), 'Belp')]

  # You may find other True Type Fonts (*.ttf) in /usr/share/fonts/truetype/...
  # Windows 10 PRO users may find more True Type Fonts in C:\Windows\Fonts\ ...
  my_points = {'font': datdrp + '/EMCdata/fonts/DejaVuSerif.ttf',
               'font_size': 20, 'points_list': my_poi_list, 'symbol': 'circle',
               'ptsize': 20, 'outline': 'black', 'fill': 'red', 'width': 5.0,
               'outline_opacity': 255,'fill_opacity': 255, 'box_outline': 'navy',
               'box_linewidth': 3.0, 'box_fill': 'gold', 'box_opacity': 255}

  # Cache is useful for speedup. Does not work with dynamic 'omerc_bb' or 'laea_bb' !!
  my_cache = {'file': datdrp + '/EMCdata/cache/DAY/' + composite, 'regenerate': False}

  # Setup indivdual overlays dictionary
  my_overlays = {}
  if ADDgrid:    # Add key grid
      my_overlays['grid'] = my_grid
  if ADDcoasts:  # Add key coasts
      my_overlays['coasts'] = my_coasts
  if ADDborders: # Add key borders
      my_overlays['borders'] = my_borders
  if ADDrivers:  # Add key rivers
      my_overlays['rivers'] = my_rivers
  if ADDpoints:  # Add key points
      my_overlays['points'] = my_points
  if OVRcache:   # Add key cache
      my_overlays['cache'] = my_cache

  if area == 'swath': # Overlays impossible, save *.png as is to tmpdir
      new_scn.save_dataset(composite,satpar.satname+'.png')
  else: # Enhance image with overlay (ev. + decorate) save *.png tmpdir
      new_scn.save_dataset(composite,satpar.satname+'.png',
              overlay = {'coast_dir': datdrp + '/EMCdata/gshhg-shp',
             'overlays' : my_overlays}) # 2^6=32 combinations should do

# ****************************************************************
# NEW POST PROCESSING: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# ****************************************************************


# Use IM to annotate (maybe also -equalize) and convert to *.jpg
# www.imagemagick.org/Usage/crop/#splice or .../annotating/#labeling
# This is the syntax used -draw "text OffsetX,OffsetY 'Hello World!'"
# This annotates to the left, numbers depend on original image size
# Colors can be {dark,light}blue, etc. or rgb(R,G,B) with \( escaped
  
def magick_string(satpar,genpar,tmpdir,loc_logo,EffHeight,ofile):
    """
    It seems ImageMagick under Windows cannot cope with multiline text
    that includes \n newlines. We have to draw every line separately.
    This scales all pixel values for IM according to the image height.
    Should make it easier to use annotation and logos at the left side.
    This has been optimized using MSG4 full scan with 3712x3712 pixels.
    """
    # ImageMagick (IM) postprocessing, added logos
    logo1 = loc_logo + '/'+satpar.logo
    logo2 = loc_logo + '/PyTROLL_400x400.jpg'
    
    OptHeight = 3712
    OptSplice = 680
    OptLogo1s = 540
    OptLogo1x = 50
    OptLogo1y = 40
    OptLogo2s = 530
    OptLogo2x = 55
    OptLogo2y = 55
    OptPoints = 90

    fac = EffHeight/OptHeight/2
    EffSplice = str(int(fac*OptSplice+0.5))
    EffLogo1s = str(int(fac*OptLogo1s+0.5))
    EffLogo1x = str(int(fac*OptLogo1x+0.5))
    EffLogo1y = str(int(fac*OptLogo1y+0.5))
    EffLogo2s = str(int(fac*OptLogo2s+0.5))
    EffLogo2x = str(int(fac*OptLogo2x+0.5))
    EffLogo2y = str(int(fac*OptLogo2y+0.5))
    EffPoints = str(int(fac*OptPoints+0.5))

    satname=satpar.satname

    dt = datetime.fromtimestamp(satpar.timmax)
    tmax = dt.strftime('%H:%M UTC')
    dmax = dt.strftime('%Y/%m/%d')
    if (genpar.day_pass):
      day_night='DAY'
    else:
      day_night='NIGHT'

    legend = [' \'DATE: ('+day_night+')\'\" ' , ' \''+dmax+'\'\" ',
              ' \'TIME:\'\" '                 , ' \''+tmax+'\'\" ',
              ' \'SOURCE:\'\" '               , ' \'EUMETCast\'\" ',
              ' \'SERVICE:\'\" '              , ' \''+satpar.tsrvc+'\'\" ',
              ' \'CHANNEL:\'\" '              , ' \''+satpar.rchan+'\'\" ',
              ' \'RECEIVER:\'\" '             , ' \''+genpar.receiver+'\'\" ',
              ' \'SATELLITE:\'\" '            , ' \''+satpar.satname+'\'\" ',
              ' \'SENSOR:\'\" '               , ' \''+satpar.sensor+'\'\" ',
              ' \'COMPOSITE:\'\" '            , ' \''+genpar.composite[:13]+'\'\" ',
              ' \'AREA:\'\" '                 , ' \''+genpar.area+'\'\" ',
              ' \''+genpar.composite[13:]+'\'\" '   , ' \'''\'\" ']
    multidraw = ''
    yoff = fac*470
    for n in range(0, len(legend)):
        if (((n % 2) == 0) and (n<19)):
            xoff = fac*40
            yoff += fac*150
        else:
            xoff = fac*60
            yoff += fac*100
        multidraw += '-draw \"text +'+str(int(xoff+0.5))+'+'+str(int(yoff+0.5))+legend[n]

    if (gen.opsys=='Windows'):
      bckclr='rgb(245,245,245)'
    else:
      bckclr='rgb\(245,245,245\)'

    ifile=tmpdir+'/'+satname+'.png'
    cmdstr = gen.convert_cmd+' '+ ifile+' '\
    + '-font Nimbus-Sans '\
    + '-gravity west -background '+ bckclr +' '\
    + '-splice '+EffSplice+'x0 '\
    + logo1+' -gravity northwest -geometry '+EffLogo1s+'x'\
    + EffLogo1s+'+'+EffLogo1x+'+'+EffLogo1y+' -composite '\
    + logo2+' -gravity southwest -geometry '+EffLogo2s+'x'\
    + EffLogo2s+'+'+EffLogo2x+'+'+EffLogo2y+' -composite '\
    + '-pointsize '+EffPoints+' -gravity northwest '+multidraw\
    + ofile
    return(cmdstr)

#################################################################
# Funcs to check/download historical tle's
from os import path
import configparser
import datetime as dt


#########################################################################################################
# Get TLE of 'Sat' with date 'Date'. take_one=True: one tle per day.
# Return: list of 1+2-line tle
# See https://www.space-track.org/documentation for details on REST queries
def download_tle_from_spacetrack(Sat,Date):
  from spacetrack import SpaceTrackClient
  import spacetrack.operators as op

  tlelist=[]
  configfile=toodrp + '/EMCtools/pppconfig'+ '/spacetrack.ini'
  # Provide a config file in the same directory as this file, with this format (without the # signs)
  # [configuration]
  # username = XXX
  # password = YYY
  # Use configparser package to pull in the ini file (pip install configparser)
  
  if not path.isfile(configfile):
    print_dbg("Error: configfile "+configfile+" doesn't exist. No new tle!",mestype.error)
    sys.exit()
#    return tlelist

  config = configparser.ConfigParser()
  config.read(configfile)
  configUsr = config.get("configuration","username")
  configPwd = config.get("configuration","password")
  st = SpaceTrackClient(configUsr, configPwd)
  dt1=dt.datetime.strptime(Date,'%Y-%m-%d')
  dt2=dt1+dt.timedelta(hours=23,minutes=59)
  drange = op.inclusive_range(dt1,dt2)
  try:
    lines = st.tle(object_name=Sat, iter_lines=True, epoch=drange, orderby='TLE_LINE1', format='tle')
  except:
    sys.exit("Error: Website space-track not available, cannot get tle's for "+Sat+" date "+Date)
  nrl=0
  tlelist.append(Sat+'\n')
  for line in lines:
    nrl=nrl+1
    tlelist.append(line+'\n')
    if line[0]=='2':
      break;

  if nrl<2:
    tlelist=[]

  return tlelist;

#########################################################################################################
# Read one tle-set from fp: 2+1-line. Do some recovering in case of invalid format
def get_one_tle(fp):
  stat=-1
  line0=''
  line1=''
  line2=''
  while True:
    line=fp.readline()
    if not line:
      break
    if line[0]=='1' and stat==0:
      line1=line
      stat=1
    elif line[0]=='2' and stat==1:
      line2=line
      stat=2
      break
    else:
      line0=line
      stat=0
  return stat,line0,line1,line2


#########################################################################################################
# Find tle in 'tle_file' of sat 'Sat' and closest to date 'Date'
# return: (age,tle):
#   age: actual age in days
#   tle: TLE (from tletools)
def find_tle(Sat,Date,tle_file):
  tle_lines=[]
  age=1000
  try:
    print_dbg('tle_file='+str(tle_file),mestype.info)
    with open(tle_file,'r') as fp:
      while True:
        (stat,line0,line1,line2)=get_one_tle(fp)
        if stat != 2:
          break

        if Sat.casefold() in line0.casefold() :
          # date from tle
          sline1=line1.split()
          yd=int(float(sline1[3]))

          # date requested
          dt0=dt.datetime.strptime(Date,'%Y-%m-%d')
          dt1=int(dt0.strftime('%y%j'))
          dift=abs(dt1-yd)
          if dift<age:
            age=dift
            mline0=line0
            mline1=line1
            mline2=line2
    tle_lines.append(mline0)
    tle_lines.append(mline1)
    tle_lines.append(mline2)

    tle = TLE.from_lines(*tle_lines)
  except:
    tle=None
  return tle_lines,tle,age

#########################################################################################################
# Get tle of sat 'Sat', date 'Date', not older than 'age' days, in tle-file 'tle_file'
# If not in tle_file: get a new tle-set from space-track, and add to tle_file
# age: allowed age of tle in days
# return: TLE
def get_hist_tle(Sat,Date,age=0,tle_file=''):
  loc_histtle=toodrp + '/EMCtools/pppconfig'
  tle=''
  try:
    dt0=dt.datetime.strptime(Date,'%Y-%m-%d')
  except:
    try:
      dt0=dt.datetime.strptime(Date,'%Y%m%d')
      Date=dt.datetime.strftime(dt0,'%Y-%m-%d')
    except:
      print_dbg("Error: Date '"+Date+"' wrong? Format should be YYYY-mm-dd",mestype.error)
      return [],tle,0

  if tle_file == '':
    tle_file=loc_histtle+'/'+Sat+'.tle'
    tle_file=tle_file.replace(' ','-')  
  print_dbg("Used historical tle file: " + tle_file,mestype.info)

  (tlelines,tle,act_age)=find_tle(Sat,Date,tle_file)
  if act_age>age:
    print_dbg('Time difference for '+Sat+' = '+str(act_age)+' days = too much, get better tle...',mestype.warn)
    tle_list=download_tle_from_spacetrack(Sat,Date)
    if tle_list == []:
      print_dbg("ERROR reading tle for " + Sat+"!",mestype.error)
    else:
      with open(tle_file,'a') as fp:
        for s in tle_list:
          fp.write(s)
        fp.close()

  (tlelines,tle,act_age)=find_tle(Sat,Date,tle_file)
  if act_age>age:
    print_dbg("Time difference for "+Sat+" = "+str(act_age)+" days = too much, can't get better tle...",mestype.warn)
  else:
    print_dbg("Time difference for "+Sat+" = "+str(act_age)+" days",mestype.info)
  return tlelines,tle,act_age

# Set orbit, download tle if necessary
# allowed_age: don't get new tle if tle has age of max. 'allowed_age' (in days)
def set_orbit(satpar,date,use_hist_tle,tlefil):
  from satpy_settings import max_tle_age

  if use_hist_tle:
    (tlelines,x,y)=get_hist_tle(satpar.satname_st,date,age=max_tle_age,tle_file=tlefil)
    if tlelines == []:
      orb = Orbital(satpar.satname, tlefil)
    else:
      orb = Orbital(satpar.satname, line1=tlelines[1],line2=tlelines[2])
  else:
    orb = Orbital(satpar.satname, tlefil)

  return orb

from tkinter import *
from PIL import ImageTk, Image

def satpy_viewer(fn):
  maxwidth=1000
  maxheight=1000
  root = Tk()
  root.title(fn)
  img1 = Image.open(fn)
  w1=img1.width
  h1=img1.height
  sfactx=maxwidth/w1
  sfacty=maxheight/h1
  sfact=min(sfactx,sfacty)
  w1=int(w1*sfact)
  h1=int(h1*sfact)
  img2=img1.resize((w1,h1))

  img = ImageTk.PhotoImage(img2)
  label = Label(image=img)
  label.grid()
  root.mainloop()

def open_viewer(viewer,fn):
  if viewer=='':
    satpy_viewer(fn)
  else:
    os.system(viewer + ' ' + fn)
