#!/usr/bin/env python3
####################################################################################
# satpy selector
# Hilversum   , 2023/04/01       License GPL3          (c) Rob Alblas
####################################################################################
# Needs zero or one argument which selects a path to data; see data_loc table in satpy_settings.py
# Files found in that location will be used to determine which command to use.
#   (Data from different sats may be in one location.)
#
####################################################################################
version='24-05-01'
support_plottrack=True
#
import os,sys
from os import walk
import fnmatch
from glob import glob
from datetime import datetime,timedelta
from pyorbital.orbital import Orbital

from tkinter import *
import tkinter as tk

from satpy_settings import geosattypes,leosattypes
from satpy_settings import data_loc,tlefil
from satpy_settings import composite_list
from satpy_settings import overlay_list

if support_plottrack:
  from satpy_plottrack import plot_track

class SatInfo:
  def __init__(self):
    self.name=''
    self.sid=''
    self.dtid=''
    self.is_polar=False
    self.date1=None
    self.time1=None
    self.date2=None
    self.time2=None
    self.chan=None


class SatTree:
  def __init__(self, name=None,date1=None,time1=None,fn=None,sid=None,dtid=None,sc=None,is_polar=None,date2=None,time2=None):
    self.head     = None
    self.next     = None
    self.sub      = None
    self.name     = name
    self.chan     = None
    self.date1    = date1
    self.time1    = time1
    self.date2    = date2
    self.time2    = time2
    self.fn       = fn
    self.sid      = sid
    self.dtid     = dtid
    self.sc       = sc
    self.is_polar = is_polar
    self.nr       = 1
    self.timeset  = []

  def __iter__(self):
    node = self.head
    while node is not None:
      yield node
      node = node.next

# for test only: print SatTree
def pri_tree(t):
  print('================================')
  for t1 in t:
    print(t1.name)
    for s1 in t1.sub:
      print('  '+s1.chan)

# tle file (full path!); if not filled in: default (defined in script used) (!Check age and update!)
# This file is NOT used if 'Hist tle' is ticked. 
file_tle    =""


# detect sat from filename
# return: (id,command)
def detect(fn):
  for st in geosattypes+leosattypes:
    for nn in st[1]:
      if fnmatch.fnmatch(fn,nn):
        return (st[0],st[2])
  return (' ',' ')

# add sat to sattree
def add_node(hn,name,date1,time1,fn,sid,dtid,sc,is_polar,date2,time2):
  nn=SatTree(name,date1,time1,fn,sid,dtid,sc,is_polar,date2,time2)
  if hn==None:
    hn=nn
    hn.head=hn
  else:
    n=hn
    while (n.next):
      n=n.next
    n.next=nn
  return nn

# add channel to sat in sattree
def add_sub(sn,chan):
  if sn.sub:
    for ss in sn.sub:
      if ss.chan==chan:
        ss.nr+=1
        return

  ss=SatTree()
  ss.chan=chan
  s=sn.sub
  if s==None:
    sn.sub=ss
    sn.sub.head=sn.sub
  else:
    while (s.next):
      s=s.next
    s.next=ss

# make tree with sats, date/time etc.
def make_tree(sattree,si,fn,sc):
  if sattree:
    for ss in sattree:
      if si.name == ss.name:
        if si.date1 == ss.date1:      #   same date
          if si.is_polar:             #     polar:
            if si.time1<ss.time1:
              ss.time1=si.time1       #       make start time minimum time
            if si.time2>ss.time2:
              ss.time2=si.time2       #       make stop time maximum time
            ss.nr+=1                  #       increment count
            if si.chan:
              add_sub(ss,si.chan)

            ss.timeset.append([si.time1,si.time2])
            return sattree            #       skip, don't add to table
          else:                       #     geo:
            if si.time1==ss.time1:    #       same time:
              ss.nr+=1                #         increment count
              if si.chan:
                add_sub(ss,si.chan)
              return sattree          #       skip, don't add to table
            if str(si.dtid)!='':    #       same time:
              if si.dtid==ss.dtid:    #       same time:
                ss.nr+=1                #         increment count
                if si.chan:
                  add_sub(ss,si.chan)
                return sattree          #       skip, don't add to table

  node=add_node(sattree,si.name,si.date1,si.time1,fn,si.sid,si.dtid,sc,si.is_polar,si.date2,si.time2)
  if si.chan:
    add_sub(node,si.chan)
  if sattree==None:
    sattree=node
  return sattree

# Collect MSG files
#H-000-MSG3__-MSG3_RSS____-_________-PRO______-201912121315-__ 
#H-000-MSG1__-MSG1_IODC___-IR_097___-000008___-201712170830-C_
#H-000-MSG4__-MSG4________-IR_120___-000006___-201204081230-C_
#H-000-MSG3__-MSG3_RSS____-IR_120___-000006___-201204081230-C_
#012345678901234567890123456789012345678901234567890123456789
#0         1         2         3         4         5         
#H-000-MSG3__-MSG3_RSS____-HRV___ ___-000024___-201912121310-C_  H-000-MSG4__-MSG4________-HRV______-000021___-201912121000-C_
#H-000-MSG3__-MSG3_RSS____-IR_016___-000006___-201912121000-C_  H-000-MSG4__-MSG4________-HRV______-000021___-201912121015-C_
def collect_msg(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=False
  satinfo.name=fn[13:22]
  if fn[36:39]=='PRO' or fn[36:39]=='EPI':
    satinfo.chan=fn[36:39]
  else:
    satinfo.chan=fn[26:32]
  satinfo.sid=satinfo.name[3]
  satinfo.date1=fn[46:54]
  satinfo.time1=fn[54:58]
  satinfo.date2=satinfo.date1
  satinfo.time2=satinfo.time1
  satinfo.fn=fn
  sattree=make_tree(sattree,satinfo,fn,sc)

  return sattree

# Collect MSG native files
#MSG3-SEVI-MSG15-0100-NA-20220918164418.769000000Z-NA.nat
#01234567890123456789012345678901234567890123456789012345
#0         1         2         3         4         5         
def collect_msgn(sattree,fn,sc):
  sat=fn[0:4]
  sid=sat[3]+'n'

  satinfo=SatInfo()
  satinfo.is_polar=False
  satinfo.name=sat
  
  satinfo.sid=sid
  satinfo.date1=fn[24:32]
  satinfo.time1=fn[32:36]
  satinfo.date2=satinfo.date1
  satinfo.time2=satinfo.time1

  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree


# Collect MTG files
#W_XX-EUMETSAT-Darmstadt,IMG+SAT,MTI1+FCI-1C-RRAD-FDHSI-FD--CHK-BODY---NC4E_C_EUMT_20170920120411_GTT_DEV_20170920115849_20170920115911_N__T_0072_0038.nc
#01234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
#0         1         2         3         4         5         6         7         8         9         0         1         2         3         4         5
#W_XX-EUMETSAT-Darmstadt,IMG+SAT,MTI1+FCI-1C-RRAD-FDHSI-FD--CHK-BODY--DIS-NC4E_C_EUMT_20231221111033_IDPFI_OPE_20231221110908_20231221110924_N_JLS_C_0067_0040.nc
def collect_mtg(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=False
  satinfo.name=fn[32:36]
#  date=fn[82:90]
#  time=fn[90:94]
  date=fn[110:118]
  time=fn[118:121]+"0"
  
  dtid=fn[148:152]
  satinfo.sid=satinfo.name[3]
  satinfo.dtid=dtid
  dt=datetime(int(date[0:4]),int(date[4:6]),int(date[6:8]),int(time[0:2]),int(time[2:4]))
  satinfo.date1=dt.strftime('%Y%m%d')
  satinfo.time1=dt.strftime('%H%M')
  satinfo.date2=satinfo.date1
  satinfo.time2=satinfo.time1
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect GOES files
#OR_ABI-L1b-RadF-M6C10_G16_s20220332350207_e20220332359526_c20220332359574.nc
#012345678901234567890123456789012345678901234567890123456789
#0         1         2         3         4         5
def collect_goes(sattree,fn,sc):
  sat='GOES'+fn[23:25]
  dt=datetime.strptime(fn[27:34],"%Y%j")
  satinfo=SatInfo()
  satinfo.name=sat
  satinfo.is_polar=False
  satinfo.date1=dt.strftime("%Y%m%d")
  satinfo.time1=fn[34:38]
  satinfo.date2=satinfo.date1
  satinfo.time2=satinfo.time1
  satinfo.sid=sat[4:6]
  satinfo.chan=fn[18:21]
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect Himawati files
#IMG_DK01VIS_201809130339_010.bz2
#01234567890123456789012345678901
#0         1         2         3 
def collect_hima(sattree,fn,sc):
  sat='HIMA8'
  satinfo=SatInfo()
  satinfo.name=sat
  satinfo.is_polar=False
  satinfo.date1=fn[12:20]
  satinfo.time1=fn[20:24]
  satinfo.date2=satinfo.date1
  satinfo.time2=satinfo.time1
  satinfo.sid=sat[4]
  satinfo.chan=fn[8:11]
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect METOP files
#AVHR_xxx_1B_M03_20220407094003Z_20220407094303Z_N_O_20220407104912Z.bz2
#AVHR_xxx_1B_M03_20220407094303Z_20220407094603Z_N_O_20220407104943Z.bz2
#01234567890123456789012345678901234567890123456789012345678901234567890
#          1         2         3         4         5         6         7
def collect_metop(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True
  if fn[14]=='1':
    satinfo.name='METOP-B'
  elif  fn[14]=='2':
    satinfo.name='METOP-A'
  elif  fn[14]=='3':
    satinfo.name='METOP-C'
  else:
    satinfo.name='METOP-?'

  satinfo.sid=satinfo.name[6]
  satinfo.date1=fn[16:24]
  satinfo.time1=fn[24:30]
  satinfo.date2=fn[32:40]
  satinfo.time2=fn[40:46]

  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect NOAA/Suomi files
#SVMC_j01_d20220215_t1359144_e1400372_b21994_c20220215141054000943_eum_ops.h5
#SVDNBC_j01_d20220215_t1042555_e1044200_b21992_c20220215105650000435_eum_ops.h5
#SVMC_npp_d20151119_t22*eum_ops.h5
#01234567890123456789012345678901234567890123456789012345678901234567890
#          1         2         3         4         5         6         7
def collect_noaa(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True

  if fn.find('_npp_',0)>0:
    satinfo.name='Suomi-NPP'
    satinfo.sid='SUO'
  else:
    satinfo.name='NOAA-20'
    satinfo.sid='N20'

  dpos=fn.find('_d',0)+2
  tpos=fn.find('_t',0)+2
  epos=fn.find('_e',0)+2
  satinfo.date1=fn[dpos:dpos+8]
  satinfo.time1=fn[tpos:tpos+6]
  satinfo.date2=satinfo.date1
  satinfo.time2=fn[epos:epos+6]
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree


# Collect FY3D files
#FY3D_20220302_215300_215400_22265_MERSI_GEO1K_L1B.HDF
#012345678901234567890123456789012345678901234567890123
#          1         2         3         4         5   
def collect_fy3d(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True
  satinfo.name='FY-3D'
  satinfo.sid='FY-3D'
  satinfo.date1=fn[5:13]
  satinfo.time1=fn[14:20]
  satinfo.date2=satinfo.date1
  satinfo.time2=fn[21:27]
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect Aqua files
#thin_MOD021KM.A2022069.1500.061.2022069162753.NRT.hdf
#thin_MYD021KM.A2022069.1535.061.2022069165545.NRT.hdf
#012345678901234567890123456789012345678901234567890123
#          1         2         3         4         5   
def collect_aqua(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True
  satinfo.name='EOS-Aqua'
  satinfo.sid=satinfo.name[0]

  dt1=datetime.strptime(fn[15:22]+fn[23:27],"%Y%j%H%M")
  satinfo.date1=dt1.strftime("%Y%m%d")
  satinfo.time1=dt1.strftime("%H%M")

  dt2=dt1+timedelta(minutes=5)
  satinfo.date2=dt2.strftime("%Y%m%d")
  satinfo.time2=dt2.strftime("%H%M")

  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree


# Collect Terra files
def collect_terra(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True
  satinfo.name='EOS-Terra'
  satinfo.sid=satinfo.name[0]

  dt1=datetime.strptime(fn[15:22]+fn[23:27],"%Y%j%H%M")
  satinfo.date1=dt1.strftime("%Y%m%d")
  satinfo.time1=dt1.strftime("%H%M")

  dt2=dt1+timedelta(minutes=5)
  satinfo.date2=dt2.strftime("%Y%m%d")
  satinfo.time2=dt2.strftime("%H%M")

  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect Sentinel files
#S3B_OL_1_EFR____20220310T091806_20220310T092106_20220311T143704_0179_063_264_2340_MAR_O_NT_002.SEN3.tar
#012345678901234567890123456789012345678901234567890123
#          1         2         3         4         5   
def collect_sentinel(sattree,fn,sc):
  satinfo=SatInfo()
  satinfo.is_polar=True
  satinfo.name='Sentinel-'+fn[1:3]
  satinfo.sid=fn[2]
  satinfo.date1=fn[16:24]
  satinfo.time1=fn[25:31]
  satinfo.date2=satinfo.date1
  satinfo.time2=fn[41:46]
  sattree=make_tree(sattree,satinfo,fn,sc)
  return sattree

# Collect all files present in location 'loc_src'.
# Return: sattree
def collect_files(loc_src):
  # Get files from loc_src
  _, _, filenames = next(walk(loc_src), (None, None, []))
  sattree=None
  st=None
  for fn in filenames:
    (nn,sc)=detect(fn)
    if nn=='MSG':
      st=collect_msg(sattree,fn,sc)
    elif nn=='MSGn':
      st=collect_msgn(sattree,fn,sc)
    elif nn=='MTG':
      st=collect_mtg(sattree,fn,sc)
    elif nn=='GOES':
      st=collect_goes(sattree,fn,sc)
    elif nn=='HIMA':
      st=collect_hima(sattree,fn,sc)
    elif nn=='Metop':
      st=collect_metop(sattree,fn,sc)
    elif nn=='NOAA':
      st=collect_noaa(sattree,fn,sc)
    elif nn=='SUOMI':
      st=collect_noaa(sattree,fn,sc)
    elif nn=='FY3D':
      st=collect_fy3d(sattree,fn,sc)
    elif nn=='Aqua':
      st=collect_aqua(sattree,fn,sc)
    elif nn=='Terra':
      st=collect_terra(sattree,fn,sc)
    elif nn=='Sentinel':
      st=collect_sentinel(sattree,fn,sc)
    elif nn==None:
      None
    else:
      None

    if sattree==None:
      sattree=st
    
  if sattree:
    sattree=sorted(sattree,key=lambda sattree:sattree.name+sattree.date1+sattree.time1)
    for st in sattree:
      sub=st.sub
      if sub:
        st.sub=sorted(sub,key=lambda sub:sub.chan)

      ts=st.timeset
      if ts!=[]:
        st.timeset=sorted(ts,key=lambda ts:ts)
  return sattree


# Fill gui tree with sattree
def fill_tree(sattree,tree):
  if sattree==None:
    return
  for item in sattree:
    itemx=[]
    itemx.append(item.name)
    itemx.append(item.date1)
    itemx.append(item.time1)
    itemx.append(item.nr)
    itemx.append(item.fn)        # col=5       hidden: filename
    itemx.append(item.sid)       # col=5       hidden: satid
    itemx.append(item.sc)        # col=6       hidden: script
    itemx.append(item.is_polar)  # col=7       hidden: polar flag
    itemx.append(item.date2)     # col=8       hidden: date2
    itemx.append(item.time2)     # col=9       hidden: time2
    itemx.append(item.timeset)           # col=10       hidden: time2
    tid=tree.insert('','end',values=itemx)

    nrhrv=0
    if item.sub:
      for i2 in item.sub:
        itemx=[]
        itemx.append(i2.chan)
        itemx.append(item.date1)
        itemx.append(item.time1)
        itemx.append(i2.nr)
        itemx.append(item.fn)        # col=5       hidden: filename
        itemx.append(item.sid)       # col=5       hidden: satid
        itemx.append(item.sc)        # col=6       hidden: script
        tree.insert(tid,'end',values=itemx)

# Convert string 'True','False' to boolean
def str2bool(s):
  if s=='True':
    return True
  if s=='False':
    return False
#  print('ERR str2bool: >'+str(s)+'<')
  return False
  
####################################################################################
# Start gui part
####################################################################################
# Handle message
def set_message(w,s,clr=''):
  if clr=='':
    clr='Black'
  w.delete(0,END)
  if (len(s)>=1):
    w.insert(0,s)
    w.config(fg=clr)

def ovl2str():
  ovlstr=''
  for item in overlay_list:
    val=ovl[item[0]].get()
    if val:
      if item[0]=="default":
        ovlstr=item[0];
        break;
      else:
        if item[0]=="cities":
          ovlstr=ovlstr+'C'
        else:
          ovlstr=ovlstr+item[0][0]
  if ovlstr=='':
    ovlstr='None'
  return ovlstr

####################################################################################
# Create the gui
sel_area=''
sel_comp=''
ovl={}
sel_ovl=[]
col_polarflag=7
def setup_gui(sattree,sel_sat):
  import tkinter.ttk as ttk
  import tkinter.font as tkFont
  import tkinter.messagebox as mb
  from satpy_areas import collect_all_areas,create_regioarea,regio_list
  global entry1
  global area
  global sel_area
  global sel_comp
  global sel_ovl
  global ovl
  def handle_keypress_view(event):
    items = tree.focus()
    stype=tree.set(items,column=0)
    date=tree.set(items,column=1)
    time=tree.set(items,column=2)
    # nr=column 3
    # fn=column 4
    sid=tree.set(items,column=5)
    sc=tree.set(items,column=6)
    is_polar=str2bool(tree.set(items,column=col_polarflag))
    if is_polar:
      time=''

    doit(wnd,stype,date,time,sc,True,sid,day_night.get(),histtle.get(),stacked.get())

  def handle_keypress_gen(event):
    items = tree.focus()
    stype=tree.set(items,column=0)
    date=tree.set(items,column=1)
    time=tree.set(items,column=2)
    # nr=column 3
    # fn=column 4
    sid=tree.set(items,column=5)
    sc=tree.set(items,column=6)
    is_polar=str2bool(tree.set(items,column=col_polarflag))
    if is_polar:
      time=''
    doit(wnd,stype,date,time,sc,False,sid,day_night.get(),histtle.get(),stacked.get())

  def do_track(is_polar):
    if support_plottrack:
      items = tree.focus()
      if items=='':
        return
      is_polar=str2bool(tree.set(items,column=col_polarflag))
      if is_polar:
        satname=tree.set(items,column=0)
        date1=tree.set(items,column=1)
        time1=tree.set(items,column=2)
        date2=tree.set(items,column=8)
        time2=tree.set(items,column=9)
        timeset=tree.set(items,column=10)
        set_message(entry1,"Plotting...")
        wnd.update()
        plot_track(satname,date1,time1,date2,time2,timeset,sel_area,tlefil,histtle.get())
        set_message(entry1,"Ready")
        wnd.update()
      else:
        plot_track() # remove track window

  def handle_keypress_track(event):
    do_track(True)

  def do_selitem():
    set_message(entry1,"Loading...")
    wnd.update()
    si=sel_item.get()
    tree.delete(*tree.get_children())
    #hierna vullen
    load_item(si,tree)
    cmenu.delete(1,"end")
    for n in data_loc:
      if  n[1][0] == si:
        mk_compmenu(cmenu,n[0])
        break
    set_message(entry1,"Ready")

  # make composite menu
  def mk_compmenu(cmenu,item):
    for st in composite_list:
      n=st[0].split('/')
      if n[0]==item:
        if len(n)==1:
          for c in st[1]:
            cmenu.add_radiobutton(value=c, label=c, variable=sel_comp)
        else:
          c_submenu = tk.Menu(cmenu, tearoff=True)
          cmenu.add_cascade(label=n[1], menu=c_submenu)
          for c in st[1]:
            c_submenu.add_radiobutton(value=c, label=c, variable=sel_comp)
        sel_comp.set(st[1][0])


  def mk_ovlmenu(omenu):
    for item in overlay_list:
      ovl[item[0]] = BooleanVar()

    def get_checked_ovl(event=None):
      ovlstr=ovl2str()
      sel_ovl.set(ovlstr)

    for item in overlay_list:
      omenu.add_checkbutton(label=item[0], variable=ovl[item[0]],command=get_checked_ovl)
      ovl[item[0]].set(item[1])

  def select_item(s=None):
    items = tree.focus()
    if items!='':
      is_polar=str2bool(tree.set(items,column=col_polarflag))
      if is_polar:
        if False:
          buttr.config(state=NORMAL)
        butd.config(state=NORMAL)
        butn.config(state=NORMAL)
        butht.config(state=NORMAL)
        butmp.config(state=NORMAL)
      else:
        if False:
          buttr.config(state=DISABLED)
        butd.config(state=DISABLED)
        butn.config(state=DISABLED)
        butht.config(state=DISABLED)
        butmp.config(state=DISABLED)

      do_track(is_polar)

  # columns
  col_vg =1
  col_sat=2
  col_are=3
  col_com=4
  col_ovl=5
  col_dn =6
  col_hs =7
  col_inf=8

  header = ['sat/chan', 'date','time','# files','fn','sattype','script ','polar ','date2','time2','ts']
  wnd = tk.Tk()
  wnd.title('Satpy selector')
  frame = ttk.Frame()
  frame.pack(fill='both', expand=True)
  tree = ttk.Treeview(columns=header, show="tree headings")
  vsb = ttk.Scrollbar(orient="vertical",command=tree.yview)
  hsb = ttk.Scrollbar(orient="horizontal",command=tree.xview)
  tree.configure(yscrollcommand=vsb.set,xscrollcommand=hsb.set);
  tree.column('#0',width=20)
  tree.column('#0',stretch=0)
  tree.bind('<ButtonRelease-1>', select_item)


  tree.grid(column=0, row=0, sticky='nsew', in_=frame)
  vsb.grid(column=1, row=0, sticky='ns', in_=frame)
  hsb.grid(column=0, row=1, sticky='ew', in_=frame)
  frame.grid_columnconfigure(0, weight=1)
  frame.grid_rowconfigure(0, weight=1)

  frame1=Frame(wnd)


  but1=tk.Button(frame1,text="View")
  but1.bind("<Button-1>",handle_keypress_view)
  but1.grid(column=col_vg,row=1)

  but2=tk.Button(frame1,text="Gen.")
  but2.bind("<Button-1>",handle_keypress_gen)
  but2.grid(column=col_vg,row=2)

# No button plottrack, done by selecting sat.
  if False:
   if support_plottrack:
    buttr=tk.Button(frame1,text="track")
    buttr.bind("<Button-1>",handle_keypress_track)
    buttr.grid(column=col_vg,row=3)

  day_night = IntVar()
  butd=Radiobutton(frame1,text='Day  ',value=0,variable=day_night)
  butn=Radiobutton(frame1,text='Night',value=1,variable=day_night)
  butd.grid(column=col_dn,row=1)
  butn.grid(column=col_dn,row=2)

  histtle = IntVar()
  butht=Checkbutton(frame1,text='Hist tle',onvalue=1,variable=histtle)
  butht.grid(column=col_hs,row=1)
  histtle.set(1) # set default 'on'

  stacked = IntVar()
  butmp=Checkbutton(frame1,text='MPass',onvalue=1,variable=stacked)
  butmp.grid(column=col_hs,row=2)

  ##### create item menu #####
  slabel=tk.Label(frame1,text='Satellite')
  sel_item = tk.StringVar(value=sel_sat) # data_loc[0][0])
  smenubutton = tk.Menubutton(frame1, textvariable=sel_item ,indicatoron=True,borderwidth=1, relief="raised")
  smenu = tk.Menu(smenubutton, tearoff=True)
  smenubutton.configure(menu=smenu)
  for dl in data_loc:
    smenu.add_radiobutton(value=dl[1][0], label=dl[1][0], variable=sel_item,command=do_selitem)

  smenu=smenubutton
  slabel.grid(column=col_sat,row=1)
  smenu.grid(column=col_sat,row=2)

  ##### create area menu #####
  areas=collect_all_areas()
  regio_area=create_regioarea(regio_list,areas)
  if regio_area == None:
    alabel=tk.Label(frame1,text='No Area')
  else:
    alabel=tk.Label(frame1,text='Area')

  sel_area = tk.StringVar(value="default")
  amenubutton = tk.Menubutton(frame1, textvariable=sel_area, indicatoron=True,borderwidth=1, relief="raised")
  amenu = tk.Menu(amenubutton, tearoff=True)
  amenubutton.configure(menu=amenu)

  if regio_area != None:
    # Create 2D area menu
    for p in regio_area:
      a_submenu = tk.Menu(amenu, tearoff=True)
      amenu.add_cascade(label=p.name, menu=a_submenu)
      for q in p.area:
        a_submenu.add_radiobutton(value=q.name, label=q.name, variable=sel_area,command=select_item)
  amenu=amenubutton

  alabel.grid(column=col_are,row=1)
  amenu.grid(column=col_are,row=2)

  ##### create composite menu #####
  clabel=tk.Label(frame1,text='Composite')
  sel_comp = tk.StringVar(value="default")
  cmenubutton = tk.Menubutton(frame1, textvariable=sel_comp, indicatoron=True,borderwidth=1, relief="raised")
  cmenu = tk.Menu(cmenubutton, tearoff=True)
  cmenubutton.configure(menu=cmenu)
  mk_compmenu(cmenu,data_loc[0][0])
  sel_comp.set('default')

  clabel.grid(column=col_com,row=1)
  cmenubutton.grid(column=col_com,row=2)

  ##### create overlay menu #####
  olabel=tk.Label(frame1,text='Overlays')
  sel_ovl = tk.StringVar(value="")
  omenubutton = tk.Menubutton(frame1, textvariable=sel_ovl, indicatoron=True,borderwidth=1, relief="raised")
  omenu = tk.Menu(omenubutton, tearoff=True)
  omenubutton.configure(menu=omenu)
  mk_ovlmenu(omenu)
  olabel.grid(column=col_ovl,row=1)
  omenubutton.grid(column=col_ovl,row=2)

  sel_ovl.set(ovl2str())


  ##### info entry #####
  elabel=tk.Label(frame1,text='Info')
  entry1=Entry(frame1,font=('Arial', 12, 'bold'),width=9)
  elabel.grid(column=col_inf,row=1)
  entry1.grid(column=col_inf,row=2)

  set_message(entry1,'V '+version)

  frame1.pack(side=tk.TOP,fill="x")

  tree["displaycolumns"]=(header[0],header[1],header[2],header[3])

  for col in header:
    tree.heading(col, text=col.title())
    tree.column(col,width=tkFont.Font().measure(col.title()))
 
  fill_tree(sattree,tree)

  wnd.mainloop()


#loc_src repareren!
loc_src=''
def load_item(satid,tree):
  global loc_src
  loc_src=''
  for dtype in data_loc:
    if satid == dtype[1][0]:
      if dtype[1][1]!='':
        loc_src=dtype[1][1]
      else:
        print('No location defined for '+dtype[1][0])
      break
  if loc_src!='':
    sattree=collect_files(loc_src)
    fill_tree(sattree,tree)

# Do action
# time=='': no time -> polar, else geo
def doit(wnd,stype,date,time,script,do_view,sid,dn,do_histtle,do_stacked):
  if stype[:4]=='Sent':
    if do_stacked:
      set_message(entry1,"No MPass!","Red")
      return

  if time=='':
    if dn==0:
      sidn=sid+' -dn D'
    else:
      sidn=sid+' -dn N'
  else:
    sidn=sid
  
  if sidn=='':
    cmd=script + ' -t '+ date+time+ ' -src ' +loc_src
  else:
    cmd=script + ' -t '+ date+time+' -sat ' + sidn + ' -src ' +loc_src

  if time=='':
    if file_tle!='':
      cmd=cmd+' -tle '+ file_tle

  selected_area = sel_area.get()
  if selected_area!='default' and selected_area!='':
    cmd=cmd+' -area '+str(selected_area)

  selected_comp = sel_comp.get()
  if selected_comp!='default':
    cmd=cmd+' -composite '+selected_comp

  if do_view:
    cmd=cmd+' -v'

  ovlopt=ovl2str()
  if (ovlopt=='default'):
    ovlopt='D';

  if ovlopt!='':
    cmd=cmd+' -ovl '+ ovlopt
  else:
    cmd=cmd+' -ovl N'

  #Only add -histtle and -mp option if no time spec., equals it's a polar sat.
  if time=='':
    if do_histtle:
      cmd=cmd+' -histtle'
    if do_stacked:
      cmd=cmd+' -mp'

  set_message(entry1,"Running")
  wnd.update()
  print(cmd)

  os.system(cmd)
  set_message(entry1,"Ready")

def usage(prog):
  print('usage:')
  print('  '+prog+' [options]')
  print('   [options]:')
  for n in data_loc:
    print('     {:10s} (location: {:s})'.format(n[1][0],n[1][1]))

################################################################################
# Main part
#
print('Use settings in satpy_settings.py')

sat=''
if len(sys.argv) >1:
  if sys.argv[1]=='-h':
    usage(sys.argv[0])
    quit()

if len(sys.argv) <=1:
  for dtype in data_loc:
    if dtype[1][1]!='':
      sat=dtype[1][0]
      loc_src=dtype[1][1]
      break
else:
  sat=sys.argv[1]
  for dtype in data_loc:
    if sat == dtype[1][0]:
      if dtype[1]!='':
        loc_src=dtype[1][1]
      else:
        sys.exit('No location defined for '+dtype[1][0])
      break

  if loc_src=='':
    loc_src=sys.argv[1]

print("Selected file location: " + loc_src)
if loc_src[0:1]!='/' and loc_src[1:2]!=':' :
  print("ERROR: This doesn't look like a full path! Check pathes in 'satpy_settings.py'")

# Get files from loc_src
sattree=collect_files(loc_src)
wnd=setup_gui(sattree,sat)

#if __name__ == "__main__":
#  main(sys.argv[1:])
