#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#*************************************************************************
#
# NOAA-20  JPSS (Joint Polar Satellite System)    sensor viirs M+DNB-Bands
# https://directory.eoportal.org/web/eoportal/satellite-missions/n/noaa-20
# https://www.wmo-sat.info/oscar/satellites/view/noaa_20
#
# Orbit: Sun-synchronous app. LTAN=13:30    (Local Time on Ascending Node)
#
#         ***** This script takes an argument YYYYmmddNoD *****
#                  Where NoD must be either NIG or DAY
#
# During daylight passes only MXY bands are available. During night passes
# we have DNB (moonlight and artificial illumination) plus all MXY bands!!
#***************************************************************************
#
# Suomi-NPP  JPSS (Joint Polar Satellite System)    sensor viirs M+DNB-Bands
# https://directory.eoportal.org/web/eoportal/satellite-missions/s/suomi-npp
# https://www.wmo-sat.info/oscar/satellites/view/snpp
#
# Orbit: Sun-synchronous app. LTAN=13:25      (Local Time on Ascending Node)
#
#          ***** This script takes an argument YYYYmmddNoD *****
#                  Where NoD must be either NIG or DAY
#
# During daylight passes only MXY bands are available. During night passes
# we have DNB (moonlight and artificial illumination) plus all MXY bands!!
#---------------------------------------------------------------------------
#-------------------------------------------------------------------------
#
# CH-3123 Belp, 2022/12/10        License GPL3          (c) Ernst Lobsiger
#
#*************************************************************************
#
# File Pattern : SVMC_{platform_shortname}_d{start_time:%Y%m%d_t%H%M%S%f}_e{end_time:%H%M%S%f}_b{orbit:5d}_c{creation_time:%Y%m%d%H%M%S%f}_eum_ops.h5
# Example name : SVMC_j01_d20190411_t0224358_e0226003_b07218_c20190411023355000121_eum_ops.h5
# Hugo's ruler : 012345678901234567890123456789012345678901234567890123456789012345678901234567890
# copied EMCV  : 0         1         2         3         4         5         6         7         8
# File Pattern : SVDNBC_{platform_shortname}_d{start_time:%Y%m%d_t%H%M%S%f}_e{end_time:%H%M%S%f}_b{orbit:5d}_c{creation_time:%Y%m%d%H%M%S%f}_eum_ops.h5
# Example name : SVDNBC_j01_d20190411_t0424054_e0425281_b07219_c20190411043053000049_eum_ops.h5
# Hugo's ruler : 012345678901234567890123456789012345678901234567890123456789012345678901234567890
# copied EMCV  : 0         1         2         3         4         5         6         7         8
# EUMETCast    : These files arrive as above in channel E1B-RDS-2 (Basic Service)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation
#
#***************************************************************************
#
# File Pattern : SVMC_{platform_shortname}_d{start_time:%Y%m%d_t%H%M%S%f}_e{end_time:%H%M%S%f}_b{orbit:5d}_c{creation_time:%Y%m%d%H%M%S%f}_eum_ops.h5
# Example name : SVMC_npp_d20190326_t0147221_e0148463_b38388_c20190326015546000177_eum_ops.h5
# Hugo's ruler : 012345678901234567890123456789012345678901234567890123456789012345678901234567890
# copied EMCV  : 0         1         2         3         4         5         6         7         8
# File Pattern : SVDNBC_{platform_shortname}_d{start_time:%Y%m%d_t%H%M%S%f}_e{end_time:%H%M%S%f}_b{orbit:5d}_c{creation_time:%Y%m%d%H%M%S%f}_eum_ops.h5
# Example name : SVDNBC_npp_d20190325_t0211594_e0213236_b38373_c20190325022115000637_eum_ops.h5
# Hugo's ruler : 012345678901234567890123456789012345678901234567890123456789012345678901234567890
# copied EMCV  : 0         1         2         3         4         5         6         7         8
# EUMETCast    : These files arrive as above in channel E1H-RDS-1 (HVS-1)
# ANNOTATION   : This script uses ImageMagick (IM) convert for annotation

# I need
from LEOstuff import test_argv, leo_images, get_magick
from LEO_extrastuff import get_args_leo,split_Dat,recreate_cmd,test_file_exist,satpy_viewer
import os, sys, platform

# Why to hell is it not working?
# from satpy.utils import debug_on
# debug_on()

# What you should know
OS = platform.system()

opts = get_args_leo('NOAA','SU,20,21')

Yea, Mon, Day, Hou, Min=split_Dat(opts.Date)

#Allow SU<omi> or N<nr> = <nr>
if opts.sat[:2]=='SU':
  sat='SNPP'
elif opts.sat[0]=='N':
  sat='NOAA'+str(opts.sat[1:])
else:
  sat='NOAA'+str(opts.sat)

curdir=os.getcwd()

# vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
# ********** ADAPT THESE PARAMETERS TO YOUR PERSONAL NEEDS ***********

# Edit 0-4 parameter(s) below according to your file layout:
# My files are in a /MountPoint/Channel/YYYY/mm/dd structure
# Your segdir may be named differently from EUMETSAT channel
# No trailing / because this will make timestamps unreadable

chan='E1B-RDS-2'
if opts.src != '':
  segdir = opts.src
  if segdir[-1]=='/':
    segdir=segdir[:-1]
else:
  if OS == 'Linux':
      segdir = '/srv/'+chan
  elif OS == 'Windows':
      segdir = 'Z:/'+chan
  else:
      sys.exit('Sorry, OS ' + OS + ' seems unsupported yet ...')

if opts.bulk==None:      # use default value
  isbulk = True
else:
  isbulk = opts.bulk

decomp = False

# Your TC receiver type
receiver = 'SR1'


# Available DAY composites: See files ../satpy/etc/composites/visir.yaml and viirs.yaml
# *************************************************************************************
# ['ash', 'cloudtop_daytime', 'dust', 'false_color', 'fire_temperature',
#  'fire_temperature_39refl', 'fire_temperature_awips', 'fire_temperature_eumetsat',
#  'fog', 'ir108_3d', 'ir_cloud_day', 'natural_color', 'natural_color_sun_lowres',
#  'natural_with_night_fog', 'night_fog', 'ocean_color', 'overview', 'snow_age',
#  'snow_lowres', 'true_color', 'true_color_crefl', 'true_color_lowres',
#  'true_color_lowres_crefl', 'true_color_lowres_land',
#  'true_color_lowres_marine_tropical', 'true_color_raw']
# NIG composites for DAY passes will not work if they use DNB channels

# Available NIG composites: See files ../satpy/etc/composites/visir.yaml and viirs.yaml
# *************************************************************************************
# ['adaptive_dnb', 'ash', 'dust', 'dynamic_dnb', 'fog', 'histogram_dnb', 'hncc_dnb',
#  'ir108_3d', 'natural_with_night_fog', 'night_fog', 'night_overview']
# Most DAY composites for NIG passes will be black, use DNB for Auroras

# Choose your composites for DAY and NIG passes. These must be lists containing at least one entry.
if opts.composites=='':
  if opts.NoD == 'DAY':
      composites = ['cloudtop_daytime', 'natural_color', 'true_color']
  else:   # 'NIG'
      composites = ['ir108_3d', 'night_fog', 'night_overview']
else:
  composites=opts.composites.split(',')

kwargs = {'histtle':opts.histtle}
if opts.area=='':
  # Choose your 'area' name from file areas.yaml, there is no fake area 'swath' as in Starter Kit 3.0
  # For dynamic areas 'omerc_bb', 'laea_bb' set your POI with kwargs, else POI defaults to your station
  # kwargs = {'lon': 7.0, 'lat': 47.0, 'ran': 15.0}   <--- This is an example, kwargs is a dictionary
  area = 'westminster'
else:
  area=opts.area

print('Area     : '+str(area))
print('Composite: '+str(composites))

# Depending on the size of your area you may want to display multiple passes (e.g. for area = 'eurol')
# You cannot choose multi = True for areas 'omerc_bb' and 'laea_bb' that work for single passes only!!
multi = opts.mp

# Optionally you can define an area dependant cities-list. If empty a sat default cities-list is used.
# Be aware that adding one city name will change your OVRcache even if the city is outside your image.
area_cities = []

# Configure your individual overlays, either True/False or 1/0 do work
if opts.default:               # default overlays, no -ovl option
  ADDcoasts  = True
  ADDborders = True
  ADDrivers  = True
  ADDlakes   = True
  ADDgrid    = True
  ADDcities  = True
  ADDpoints  = True
  ADDstation = True
  ADDlegend  = True
else:                          # take overlays from option -ovl
  ADDcoasts  = opts.coasts
  ADDborders = opts.borders
  ADDrivers  = opts.rivers
  ADDlakes   = opts.lakes
  ADDgrid    = opts.grid
  ADDcities  = opts.cities
  ADDpoints  = opts.points
  ADDstation = opts.station
  ADDlegend  = opts.legend

# OVRCache speeds up static overlays. These are stored in your .../SPSdata/cache
# Never use this with homebrew or SatPy's dynamic areas like omec_bb and laea_bb
# With caching you will sometimes have to delete *.png OVRcache files manually !
OVRcache   =  True

# If you are testing interactively while PyTROLL/SatPy image generation croned
# This allows for one interactively started script that will use tmpdirs/xtest
testrun = False

subdir = opts.NoD

#Only if filename is given: test if file exists, don't overwrite
if opts.ofile!='':
  pfn=test_file_exist(sat,opts.dst,subdir,opts.ofile)
  if pfn != '':
    print(pfn+' exists, nothing done.')
    quit()

# ******** TOUCH THE CODE BELOW ONLY IF YOU KNOW WHAT YOU DO *********
# ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^


Yea, Mon, Day, Hou, Min, height = leo_images(Yea, Mon, Day, sat, opts.NoD, multi, segdir, True, isbulk, 'viirs_compact',
                                             composites, area, area_cities, ADDcoasts, ADDborders, ADDrivers, ADDlakes,
                                             ADDgrid, ADDcities, ADDpoints, ADDstation, OVRcache, testrun, **kwargs)


# *******************************************************************
# POST PROCESSING LEGEND: WORKS WITH ALL SCRIPTS AND PROJECTION AREAS
# *******************************************************************

# Below is for legend only
sensor = 'viirs'
# EUMETCast Europe service
service = 'Basic'
# EUMETSAT channel naming
channel = chan
# Logos in logos directory
logo1 = 'NOAA_1200x1200.png'
logo2 = 'PyTROLL_400x400.jpg'

for composite in composites:

    magick = get_magick(Yea, Mon, Day, Hou, Min, height, logo1, logo2, service, channel, receiver,
                        sat, sensor, composite, area, testrun, opts.NoD, multi, ADDlegend,
                        subdir=subdir)  # basedir=basedir, subdir=subdir

    magick,pfn=recreate_cmd(magick,opts.dst,opts.ofile,curdir)
    # **ImageMagick**
    os.system(magick)
    print('Generated '+pfn)
    if opts.do_view:
      satpy_viewer(pfn)

# If you get an image in the satellites tmp directory but no final image with
# legend at left then you have a problem with ImageMagick. THAT'S ALL FOLKS?!
