#************************************************************************
#
# This module provides functions necessary for generating *LEO* satellite
# products using LEO funcs like Metop.py etc. containing extended options, and
# products with 'Satpy Processing System' (SPS) advanced Kit Version 4.1.
# One copy of the file must reside in the same direcory with LEO scripts.
#
#    Hilversum, 2024/04/01        License GPL3        (c) Rob Alblas
#
#************************************************************************
#
import argparse
from argparse import RawTextHelpFormatter
import platform
import os,sys

#Define viewer. If '': use internal viewer.
Viewer=''

OS = platform.system()

class options:
  def __init__(self):
    self.Date=None
    self.NoD=None
    self.mp=None
    self.histtle=None
    self.sat=None
    self.src=None
    self.bulk=None
    self.dst=None
    self.ofile=None
    self.area=None
    self.composites=None
    self.for_movie=None
    self.do_view=None

    self.ovl=None
    self.coasts  = False
    self.borders = False
    self.rivers  = False
    self.lakes   = False
    self.grid    = False
    self.cities  = False
    self.points  = False
    self.station = False
    self.legend  = False
    self.default = False

def get_args_leo(prog,sats):
  parser = argparse.ArgumentParser(prog)
  parser.add_argument('-t'        ,default=''      , help="time: YYYYmmDD")
  parser.add_argument('-dn'       ,default='D'     , help="D|N, D=Day, N=Night")
  parser.add_argument('-sat'      ,default=''      , help=sats)
  parser.add_argument('-mp'       ,nargs='?',const=True,default=False, help="do multipass")
  parser.add_argument('-histtle'  ,nargs='?',const=True,default=False, help="get historical tle from spacetrack")
  parser.add_argument('-src'      ,default=''      , help="source dir., MUST be absolute path!")
  parser.add_argument('-b'        ,default=''      , help="bulk: y |n.")
  parser.add_argument('-dst'      ,default=''      , help="destination dir.")
  parser.add_argument('-o'        ,default=''      , help="output file")
  parser.add_argument('-area'     ,default=''      , help="area, 'list' gives a list, 'list=<str>': items containing <str>")
  parser.add_argument('-composite',default=''      , help="composite, 'list' gives a list")
  parser.add_argument('-ovl'      ,default='D'     , help="overlays: c=coast,  b=borders, g=grid, \n"+\
                                                          "          r=rivers, l=lakes,   C=cities,\n"+\
                                                          "          p=points, s=station, L=Legend\n"+\
                                                          "          A=all, N=none, D or no -ovl: defaults")
  parser.add_argument('-v'        ,default=False,nargs='?',const=True, help="Start viewer")

  args = parser.parse_args()

  opts=options()

  opts.Date=args.t
  if args.dn=='N':
    opts.NoD='NIG'
  else:
    opts.NoD='DAY'

  opts.mp=args.mp
  opts.histtle=args.histtle
  opts.sat=args.sat
  opts.src=args.src
  if args.b.lower()=='y':
    opts.bulk=True
  elif  args.b.lower()=='n':
    opts.bulk=False

  opts.dst=args.dst
  opts.ofile=args.o
  opts.area=args.area
  opts.composites=args.composite
  opts.do_view=args.v

  opts.ovl        = args.ovl
  if "c" in opts.ovl or "A" in opts.ovl:
    opts.coasts  = True
  if "b" in opts.ovl or "A" in opts.ovl:
    opts.borders = True
  if "r" in opts.ovl or "A" in opts.ovl:
    opts.rivers = True
  if "l" in opts.ovl or "A" in opts.ovl:
    opts.lakes = True
  if "g" in opts.ovl or "A" in opts.ovl:
    opts.grid = True
  if "C" in opts.ovl or "A" in opts.ovl:
    opts.cities = True
  if "p" in opts.ovl or "A" in opts.ovl:
    opts.points = True
  if "x" in opts.ovl or "A" in opts.ovl:
    opts.station = True
  if "L" in opts.ovl or "A" in opts.ovl:
    opts.legend = True
  if "D" in opts.ovl:
    opts.default = True

  if opts.Date=='':
    sys.exit('Error: Option -t needed.')

  print_ovl(opts)

  return opts

#  return asat,Dat,dn,mp,histtle,src,dst,ofile,area,composite,do_view

def print_ovl(opts):
  print("Add overlays:")
  if opts.coasts:
    print("  coast")
  if opts.borders:
    print("  boarders")
  if opts.rivers:
    print("  rivers")
  if opts.lakes:
    print("  lakes")
  if opts.grid:
    print("  grid")
  if opts.cities:
    print("  cities")
  if opts.points:
    print("  points")
  if opts.station:
    print("  station")
  if opts.legend:
    print("  legend")
  if opts.default:
    print("  Default overlays")

def split_Dat(Dat):
  # Get substrings
  Yea=Dat[:4]
  Mon=Dat[4:6]
  Day=Dat[6:8]
  Hou=Dat[8:10]
  Min=Dat[10:]

  return Yea, Mon, Day, Hou, Min

# input contains: <magick_command>  
def split_ofile(magick):
  splstr=magick.rsplit(' ',1)
  prog=splstr[0]
  splstr=splstr[1].rsplit('/',1)
  loc=splstr[0]
  fn=splstr[1]
  return prog,loc,fn

def recreate_cmd(magick,dst,ofile,curdir):
  prog,loc,fn=split_ofile(magick)

  if dst!='':
    if OS == 'Linux':
      if dst[0]=='/' or dst[0]=='~':
        loc=dst
      else:
        loc=curdir+'/'+dst
    elif OS == 'Windows':      # only abs. path possible, yet
      loc=dst

  if ofile!='':
    fn=ofile

  if dst!='' or ofile!='':
    magick=prog+' '+loc+'/'+fn
  return magick,loc+'/'+fn

def test_file_exist(sat,dst,subdir,ofile):
  if dst=='':
    nn=get_prodir_areal_areaf(sat,'','')
    dst=nn[0]+'/'+subdir
  pfn=dst+'/'+ofile
  print('output='+pfn)
  if os.path.isfile(pfn):
    return pfn
  return ''

# For LEO's
#################################################################
# Funcs to check/download historical tle's
from os import path
import configparser
import datetime as dt
from LEOstuff import toodir

class mestype:
  quiet=0
  info=1
  warn=2
  error=3


def print_dbg(message,n=mestype.quiet):
  meslvl=mestype.quiet
  if n>mestype.quiet and n>=meslvl:
    print(message)

#########################################################################################################
# Get TLE of 'Sat' with date 'Date'. take_one=True: one tle per day.
# Return: list of 1+2-line tle
# See https://www.space-track.org/documentation for details on REST queries
def create_dummy_spacetrack(configfile):
  if not path.isfile(configfile):
    fp=open(configfile,"w")
    fp.write("[configuration]\n")
    fp.write("username = USERNAME\n")
    fp.write("password = PASSWORD\n")
    fp.close()

from pyorbital.orbital import Orbital
def download_tle_from_spacetrack(Sat,Date):
  try:
    from spacetrack import SpaceTrackClient
  except:
    print("No spacetrack module installed!")
    return None
  import spacetrack.operators as op
  tlelist=[]
  configfile=toodir + '/userconfig'+ '/spacetrack.ini'
  print('Using spacetrack config file '+configfile)
  # Provide a config file in the same directory as this file, with this format (without the # signs)
  # [configuration]
  # username = USERNAME
  # password = PASSWORD
  # Use configparser package to pull in the ini file (pip install configparser)
  
  if not path.isfile(configfile):
    print("Error: configfile "+configfile+" doesn't exist. Cannot use -histtle option.\nCreated dummy file.")
    create_dummy_spacetrack(configfile)
    return None

  config = configparser.ConfigParser()
  config.read(configfile)
  configUsr = config.get("configuration","username")
  configPwd = config.get("configuration","password")
  if configUsr=='USERNAME' or configPwd=='PASSWORD':
    print("Error: configfile "+configfile+" not adapted for user/password of spacetrack.")
    return None

  st = SpaceTrackClient(configUsr, configPwd)
  dt1=dt.datetime.strptime(Date,'%Y-%m-%d')
  dt2=dt1+dt.timedelta(hours=23,minutes=59)
  drange = op.inclusive_range(dt1,dt2)
  try:
    print('Sat='+str(Sat))
    lines = st.tle(object_name=Sat, iter_lines=True, epoch=drange, orderby='TLE_LINE1', format='tle')
  except:
    print("Error: Website space-track not available, cannot get tle's for "+Sat+" date "+Date)
    return None

  nrl=0
  tlelist.append(Sat+'\n')

  for line in lines:
#    print(str(nrl)+'  '+str(line))
    nrl=nrl+1
    tlelist.append(line+'\n')
    if line[0]=='2':
      break;

  if nrl<2:
    tlelist=[]

  return tlelist;

#########################################################################################################
# Read one tle-set from fp: 2+1-line. Do some recovering in case of invalid format
def get_one_tle(fp):
  stat=-1
  line0=''
  line1=''
  line2=''
  while True:
    line=fp.readline()
    if not line:
      break
    if line[0]=='1' and stat==0:
      line1=line
      stat=1
    elif line[0]=='2' and stat==1:
      line2=line
      stat=2
      break
    else:
      line0=line
      stat=0
  return stat,line0,line1,line2


#########################################################################################################
# Find tle in 'tle_file' of sat 'Sat' and closest to date 'Date'
# return: (age,tle):
#   age: actual age in days
#   tle: TLE (from tletools)
def find_tle(Sat,Date,tle_file):
  tle_lines=[]
  age=1000
  try:
    print_dbg('tle_file='+str(tle_file),mestype.info)
    with open(tle_file,'r') as fp:
      while True:
        (stat,line0,line1,line2)=get_one_tle(fp)
        if stat != 2:
          break

        if Sat.casefold() in line0.casefold() :
          # date from tle
          sline1=line1.split()
          yd=int(float(sline1[3]))

          # date requested
          dt0=dt.datetime.strptime(Date,'%Y-%m-%d')
          dt1=int(dt0.strftime('%y%j'))
          dift=abs(dt1-yd)
          if dift<age:
            age=dift
            mline0=line0
            mline1=line1
            mline2=line2
    tle_lines.append(mline0)
    tle_lines.append(mline1)
    tle_lines.append(mline2)

    tle = TLE.from_lines(*tle_lines)
  except:
    tle=None
  return tle_lines,tle,age

#########################################################################################################
# Get tle of sat 'Sat', date 'Date', not older than 'age' days, in tle-file 'tle_file'
# If not in tle_file: get a new tle-set from space-track, and add to tle_file
# age: allowed age of tle in days
# return: TLE
def get_hist_tle(Sat,Date,age=0,tle_file=''):
  loc_histtle=toodir + '/userconfig'
  tle=''
  try:
    dt0=dt.datetime.strptime(Date,'%Y-%m-%d')
  except:
    try:
      dt0=dt.datetime.strptime(Date,'%Y%m%d')
      Date=dt.datetime.strftime(dt0,'%Y-%m-%d')
    except:
      print_dbg("Error: Date '"+Date+"' wrong? Format should be YYYY-mm-dd",mestype.error)
      return [],tle,0

  if tle_file == '':
    tle_file=loc_histtle+'/'+Sat+'.tle'
    tle_file=tle_file.replace(' ','-')  
  print_dbg("Used historical tle file: " + tle_file,mestype.info)

  (tlelines,tle,act_age)=find_tle(Sat,Date,tle_file)

  if act_age>age:
    print_dbg('Time difference for '+Sat+' > '+str(act_age)+' days = too much, get better tle...',mestype.warn)
    tle_list=download_tle_from_spacetrack(Sat,Date)

    if tle_list == None or tle_list== []:
      print_dbg("ERROR downloading tle for " + Sat+"!",mestype.error)
    else:
      with open(tle_file,'a') as fp:
        for s in tle_list:
          fp.write(s)
        fp.close()

  (tlelines,tle,act_age)=find_tle(Sat,Date,tle_file)
  if act_age>age:
    print_dbg("Time difference for "+Sat+" > "+str(act_age)+" days = too much, can't get better tle...",mestype.warn)
  else:
    print_dbg("Time difference for "+Sat+" = "+str(act_age)+" days",mestype.info)
  return tlelines,tle,act_age

# Set orbit, download tle if necessary
# allowed_age: don't get new tle if tle has age of max. 'allowed_age' (in days)
# If use_hist_tle=True then tlefil not used
def set_orbit(satname,Yea, Mon, Day,tlefil,use_hist_tle=False,max_tle_age=10):
  satname_st=satname
  if satname=='EOS-Aqua':
    satname_st='Aqua'
  if satname=='EOS-Terra':
    satname_st='Terra'
  if satname=='NOAA-20':
    satname_st='NOAA 20'
  if satname=='Suomi-NPP':
    satname_st='NPP'
  if satname=='FY-3D':
    satname_st='FENGYUN 3D'
  if satname=='Sentinel-3A':
    satname_st='SENTINEL 3A'
  if satname=='Sentinel-3B':
    satname_st='SENTINEL 3B'

  date=Yea+'-'+Mon+'-'+Day
  if use_hist_tle:
    print('Get hist-tle')
    (tlelines,x,y)=get_hist_tle(satname_st,date,age=max_tle_age,tle_file=tlefil)
    if tlelines == []:
      return None
#      orb = Orbital(satname, tlefil)
    else:
      orb = Orbital(satname, line1=tlelines[1],line2=tlelines[2])
  else:
    try:
      orb = Orbital(satname, tlefil)
    except:
      print(satname+' not found in '+tlefil)
      orb=None
  return orb


from tkinter import *
from PIL import ImageTk, Image
def satpy_viewer(pfn):
  if Viewer=='':
    maxwidth=1000
    maxheight=1000
    root = Tk()
    root.title(pfn)
    img1 = Image.open(pfn)
    w1=img1.width
    h1=img1.height
    sfactx=maxwidth/w1
    sfacty=maxheight/h1
    sfact=min(sfactx,sfacty)
    w1=int(w1*sfact)
    h1=int(h1*sfact)
    img2=img1.resize((w1,h1))

    img = ImageTk.PhotoImage(img2)
    label = Label(image=img)
    label.grid()
    root.mainloop()
  else:
    os.system(Viewer+' '+pfn+'&')

