/*******************************************************************
 * RCSId: $Id: rotorfuncs_stepper.ino,v 1.2 2025/08/06 08:45:19 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content: 
 *   several motor/rotor functions for stepper motors
 *
 * public functions:
 *   run_motor_soft(ROTOR *rot,int speed)
 *   int run_motor_hard(ROTOR *rot,int speed)
 *
 * History: 
 * $Log: rotorfuncs_stepper.ino,v $
 * Revision 1.2  2025/08/06 08:45:19  ralblas
 * _
 *
 * Revision 1.1  2025/08/04 07:40:05  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#include "rotorctrl.h"

#if MOTORTYPE == MOT_STEPPER
static void moveto(ROTOR *rot)
{
  long step;
  if (!rot) return;

  step=degr2step(rot,rot->req_degr);
  CMDP(rot,moveTo(step)); // do requested
}

static boolean end_of_rot(ROTOR *rot,int speed)
{
  int end1=0,end2=0;
  boolean at_end;
  if (!rot) return false;
  if (rot->pin_end1>=0) end1=digitalRead(rot->pin_end1);
  if (rot->pin_end1>=0) end2=digitalRead(rot->pin_end2);
  at_end=(((end1) && (speed>0)) || ((end2) && (speed<0)));
  return at_end;
}

static int set_temp_maxspeed(ROTOR *rot,int speed)
{
  int maxspeed,new_speed;
  maxspeed=CMDP(rot,maxSpeed());
  new_speed=maxspeed*(float)speed/100.;
  CMDP(rot,setMaxSpeed(new_speed));              // set speed to new according to 'speed'
  return maxspeed;
}

// soft speed control (accel.)
// return: stepper: actual speed
//         DC: acc. speed from input speed
//         stepper: only acc. up, change maxspeed gives no nice accel!
int run_motor_soft(ROTOR *rot,int speed)
{
  if (!rot) return 0;

  // set speed and run (accelleration)
  if ((!speed) || (end_of_rot(rot,speed)))
  {
    CMDP(rot,setSpeed(0));            // force internally saved speed to 0
    CMDP(rot,stop());                 // force stop
    speed=0;                          //
  }
  else
  {
    int step;
    int maxspeed=set_temp_maxspeed(rot, speed);                  // set temp. maxspeed
    // set step 360 degrees from current pos. to accelerate
    step=CMDP(rot,currentPosition()+(rot->steps_degr*SIGN(speed))); 
    CMDP(rot,moveTo(step));                    // move accelerated to endpoint
    CMDP(rot,run());                           // run with accel. 
    CMDP(rot,setMaxSpeed(maxspeed));           // restore maxspeed
  }
  rot->rotated=CMDP(rot,currentPosition());
  rot->speed=speed;
  return speed;
}

// Set speed of motor directly
int run_motor_hard(ROTOR *rot,int speed)
{
  if (!rot) return 0;

  // set speed and run (without accelleration)
  if (end_of_rot(rot,speed))
  {
    CMDP(rot,setSpeed(0));          // force internally saved speed to 0
    CMDP(rot,stop());               // force stop
    speed=0;                      //
  }
  else
  {
    float nspeed;
    nspeed=CMDP(rot,maxSpeed())*(float)speed/100.;
    CMDP(rot,setSpeed(nspeed));
    CMDP(rot,runSpeed());
  }
  rot->rotated=CMDP(rot,currentPosition());
  rot->speed=speed;
  return speed;
}



// soft speed control (accel.)
// For DC motors accel. is defined by:
//   stop: #degr. to run: diff_degr
//   start: accellerate()
// For stepper: this is built-in
// return: stepper: speed detection; dc: actual speed
// end-stop for steppermotor is switch, so act directly.
// end-stop for DCmotor/pulsgiver needs some time to detect, do outside.
static int run_motor_soft_dd(ROTOR *rot)
{
  float diff_degr;
  int speed;
  if (!rot) return 0;

  diff_degr=rot->req_degr - rot->degr;
  rot->err_degr=diff_degr;

  float aspeed;
  aspeed=CMDP(rot,speed());                               // actual speed
  aspeed=(aspeed*100.) / CMDP(rot,maxSpeed());            // speed in %
  speed=(int)aspeed;                                      // now to integer

  if ((aspeed) && (!speed)) speed=(aspeed<0? -1 : +1);    // set to 1% if 0<s<1
  if (speed>100) speed=100;
  if (speed<-100) speed=-100;
  rot->speed=speed;                                       // calculated speed

  // end-stop for steppermotor is switch, so act directly.
  if (end_of_rot(rot,speed))                         // end-stop detected!
  {
    CMDP(rot,setSpeed(0));                           // force internally saved speed to 0
    CMDP(rot,stop());                                // force stop
    speed=0;
  }
  else
  {
    moveto(rot);
    CMDP(rot,run());                                 // run with accel. 
    speed=CMDP(rot,distanceToGo());                // to detect if ready, not actual speed...
  }
  rot->rotated=CMDP(rot,currentPosition());

  return speed; // DC: actual speed set (no endstop detection!), stepper: only speed detection
}
#endif

