/*******************************************************************
 * RCSId: $Id: rotorfuncs_dc.ino,v 1.2 2025/08/06 08:51:07 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content: 
 *   several motor/rotor functions for DC motors
 *
 * public functions:
 *   int run_motor_soft(ROTOR *rot,int speed)
 *   int run_motor_hard(ROTOR *rot,int speed)
 *   int run_motor_soft_dd(ROTOR *rot)
 *   boolean is_moving(ROTOR *rot)
 *
 * History: 
 * $Log: rotorfuncs_dc.ino,v $
 * Revision 1.2  2025/08/06 08:51:07  ralblas
 * _
 *
 * Revision 1.1  2025/08/04 15:18:39  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "rotorctrl.h"
extern ROTORPREFS rotprefs;

#if ((MOTORTYPE == MOT_DC_PWM) || (MOTORTYPE == MOT_DC_FIX))
// MOT_DC_PWM or MOT_DC_FIX



/*********************************************************************
 * define speed from degrees between current and target position
 * For DC motors.
 * input: deg = difference current and requested angle in degrees
 * return: speed in percents (100=max. speed)
 *********************************************************************/
static int rotor_speed(ROTOR *rot,float deg)
{
  int speed;
  float adg=fabs(deg);
  if (!rot) return 0;

  if (adg<=rot->d_degr_stop)
  {
    speed=0;
  }
  else
  {
    if (rot->ldegr_maxspd==rot->hdegr_minspd) rot->hdegr_minspd-=1;
    #if MOTORTYPE == MOT_DC_PWM
    {
      float tmp;
      tmp=(float)(adg-rot->hdegr_minspd)*(rot->maxspeed-rot->minspeed)/(float)(rot->ldegr_maxspd-rot->hdegr_minspd);
      tmp=tmp + rot->minspeed;
      speed=(int)tmp;
      speed=MIN(speed,rot->maxspeed);
      speed=MAX(speed,rot->minspeed);
    }
    #else
    { // MOTORTYPE == MOT_DC_FIX
      speed=(int)((float)(adg-rot->hdegr_minspd)*100/(float)(rot->ldegr_maxspd-rot->hdegr_minspd));
    }
    #endif

    if (deg<0) speed*=-1;
    #if SWAP_DIR
      speed*=-1;
    #endif
  }
//  rot->speed=speed;
  return speed;
}

// accelerate speed; calculate every 100 ms
static int accelspeed(ROTOR *rot,int ispeed,boolean just_incr=false)
{
  int ospeed;
  int incdec;

  static unsigned long t[2];
  int rid=(rot->id==AX_ID? 0 : 1);

  if ((just_incr) && (abs(ispeed) < abs(rot->speed))) return ispeed;

  if (millis()-t[rid] < 100) return rot->speed;
  t[rid]=millis();

  // adapt speed every 100 ms

  // ispeed=requested speed
  if (ispeed > rot->speed)
    incdec=1;
  else if (ispeed < rot->speed)
    incdec=-1;
  else
    incdec=0;

// ispeed!=0 ==> ospeed!=0!
  ospeed=rot->speed+incdec;  // increment or decrement speed

  // if under minimum speed: set to min. speed if ispeed!=0
  if (abs(ospeed)<rot->minspeed)           // speed is under minimum speed:
  {
    if ((ispeed!=0) && (rot->speed==0))    //   must still run:
      ospeed=rot->minspeed*SIGN(ispeed);   //   -> set to min. speed
    else                                   // 
      ospeed=0;                            //   otherwise stop
  }

  // limit to max. speed
  if (abs(ospeed)>rot->maxspeed) ospeed=rot->maxspeed*SIGN(ispeed);
  return ospeed;
}

// Set DC-motor speed; 'speed'=abs. value!
static void set_speed(ROTOR *rot,int speed)
{
  if (!rot) return;
  if (speed>100) speed=100;
  #if MOTORTYPE == MOT_DC_PWM
  {
    rot->pwm=(rotprefs.max_pwm*speed)/100;
    if (rot->pin_pwm>=0) pwmWrite(rot->pin_pwm,rot->pwm);
  }
  #elif MOTORTYPE==MOT_DC_FIX
  {
    rot->pwm=rotprefs.max_pwm;
    if (rot->pin_lsp>=0)
    {
      if (speed < SPEED_LOW)
      {
        digitalWrite(rot->pin_lsp, HIGH);
      }
      else
      {
        digitalWrite(rot->pin_lsp, LOW);
      }
    }
    if (rot->pin_pwm>=0) pwmWrite(rot->pin_pwm,rot->pwm);
  }
  #endif
}

// Set direction of motor
static void set_dir(ROTOR *rot,boolean dir)
{
  boolean din;
  if (!rot) return;
  if (dir!=rot->dir)
  {
    set_speed(rot,0);
    delay(10);
  }
  rot->dir=dir;

  digitalWrite(rot->pin_dir, rot->dir);
  if (rot->pin_din>=0)
  {
    digitalWrite(rot->pin_din, (rot->dir? LOW : HIGH));
  }
}


// soft speed control (accel.)
// return: stepper: actual speed
//         DC: acc. speed from input speed
//         stepper: only acc. up, change maxspeed gives no nice accel!
boolean run_motor_soft(ROTOR *rot,int pspeed)
{
  int speed;
  if (!rot) return 0;

  // set speed and run (accelleration)
  speed=accelspeed(rot,pspeed);
  if (speed)
    set_dir(rot,speed > 0? HIGH : LOW);

  set_speed(rot,abs(speed));
  rot->speed=speed;
  return (speed || pspeed);
}

// Set speed of motor directly
int run_motor_hard(ROTOR *rot,int speed)
{
  if (!rot) return 0;

  // set speed and run (without accelleration)
  if (speed)
    set_dir(rot,speed > 0? HIGH : LOW);
  set_speed(rot,abs(speed));
  rot->speed=speed;
  return speed;
}


// soft speed control (accel.)
// For DC motors accel. is defined by:
//   stop: #degr. to run: diff_degr
//   start: accellerate()
// For stepper: this is built-in
// return: stepper: speed detection; dc: actual speed
// end-stop for steppermotor is switch, so act directly.
// end-stop for DCmotor/pulsgiver needs some time to detect, do outside.
boolean run_motor_soft_dd(ROTOR *rot)
{
  float diff_degr;
  int speed,pspeed;
  if (!rot) return 0;

  diff_degr=rot->req_degr - rot->degr;
  rot->err_degr=diff_degr;


  pspeed=rotor_speed(rot,diff_degr);
  speed=accelspeed(rot,pspeed,true);
  if (speed)
    set_dir(rot,speed > 0? HIGH : LOW);
  set_speed(rot,abs(speed));
  rot->speed=speed;

  return (speed || pspeed); // DC: actual speed set (no endstop detection!), stepper: only speed detection
}

#endif

