/*******************************************************************
 * RCSId: $Id: rotorfuncs.ino,v 1.7 2025/08/06 09:34:09 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content: 
 *   several motor/rotor functions
 *
 * public functions:
 *   float to_degr(ROTOR *rot)
 *   long from_degr(ROTOR *rot)
 *   boolean convert_eastwest(GOTO_VAL *gv,GOTO_VAL *gvo)
 *   int rotor_goto(ROTOR *rot,float val)
 *   void reset_to_pos(ROTOR *rot,long pos)
 *   void run_to_pos(ROTOR *AX_rot, ROTOR *EY_rot,float ax_pos,float ey_pos,boolean relative)
 *   void run_to_endswitch(ROTOR *AX_rot, ROTOR *EY_rot,int speed)
 *   boolean in_stormpos(ROTOR *ax,ROTOR*ey)
 *
 * History: 
 * $Log: rotorfuncs.ino,v $
 * Revision 1.7  2025/08/06 09:34:09  ralblas
 * _
 *
 * Revision 1.5  2025/08/04 15:17:03  ralblas
 * _
 *
 * Revision 1.4  2025/08/03 15:24:05  ralblas
 * _
 *
 * Revision 1.2  2025/06/16 13:51:40  ralblas
 * _
 *
 * Revision 1.1  2023/08/03 08:52:29  ralblas
 * Initial revision
 *
 * Revision 1.3  2023/08/03 08:52:29  ralblas
 * _
 *
 * Revision 1.2  2023/07/21 19:12:47  ralblas
 * _
 *
 * Revision 1.1  2023/07/18 12:11:38  ralblas
 * Initial revision
 *
 * Revision 1.1  2021/07/29 08:25:38  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "rotorctrl.h"

extern ROTORPREFS rotprefs;
extern boolean wifi_connected;

// calc. degrees from step for 'rot'
float step2degr(ROTOR *rot,long step)
{
  float degr;
  if (!rot) return 0.;
  degr=(float)step*360./(float)rot->steps_degr;
  return degr;
}

// position rotor in degrees
float to_degr(ROTOR *rot)
{
  if (!rot) return 0.;
  return step2degr(rot,rot->rotated);
}

// calc. steps from degrees for 'rot'
static long degr2step(ROTOR *rot,float degr)
{
  long step;
  if (!rot) return 0.;

  step=(long)((degr*rot->steps_degr)/360);
  #if SWAP_DIR
  step*=-1;
  #endif
  return step;
}

// position rotor in steps, calculated from degrees of rot
long from_degr(ROTOR *rot)
{
  if (!rot) return 0;
  return degr2step(rot,rot->degr);
}

#if defined(USE_EASTWEST) && USE_EASTWEST
/*********************************************************************
 * For elevation/azimut, where azimut rotor cannot rotate 360 (400) degrees:
 *   use elevation rotor having 0...180 range; 
 *   determine flip-of-elevation rotor
 *
 * Input azimut: 0=north, 90=east, 180=south, 270=west
 * Input elevation 0...90
 * east pass:   0...180, azimut ranges between 340 (N-bound) and 200 (S-bound)
 * west pass: 180...360, azimut ranges between 160 (N-bound) and  20 (S-bound)
 * 
 *********************************************************************/
boolean convert_eastwest(GOTO_VAL *gv,GOTO_VAL *gvo)
{
  boolean flip_discs=false;
  *gvo=*gv;
  // force to range 0...360
  if (!gv->east_pass)
  {                           // it's a west-pass
    flip_discs=!flip_discs;
    gvo->ax=gv->ax-180.;       // substract 180 of 'goto' azimuth
    gvo->ey=180.-gv->ey;       // flip 'goto' elevation
  }
  if (gvo->ax > 360.) gvo->ax-=360.;
  if (gvo->ax <   0.) gvo->ax+=360.;

  if (ALLOW_AUTOPASSCORRECT == true)
  {
    // Check if azimut rotor is out-of-range, shouldn't happen with correct east/west pass info
    if ((gvo->ax > rotprefs.set_azim_max) && (gvo->ax < rotprefs.set_azim_min))
    {                            // correct
      flip_discs=!flip_discs;
      gvo->ax=gvo->ax-180.;        // azimuth
      gvo->ey=180.-gvo->ey;        // elevation
    }
  }

  // force to range 0...360
  if (gvo->ax > 360.) gvo->ax-=360.;
  if (gvo->ax <   0.) gvo->ax+=360.;

  return flip_discs;
}
#endif


boolean is_moving(ROTOR *rot)
{
  boolean running=true;
  if (!rot) return false;
  if (rot->rotated!=rot->pre_rotated)
  {
    rot->timeout=millis();
  }
  else if (millis()-rot->timeout > PLS_TIMEOUT)
  {
    running=false;
  }
  rot->pre_rotated=rot->rotated;
  if (!running) rot->speed=0;
  return running;
}



/*********************************************************************
 * Rotor to angle 'val'.
 * Must be used in loop; each time this func. is executed 
 *   speed is determined and used.
 * return: current speed; 0=stop=rotator is at requested position.
 *********************************************************************/
boolean rotor_goto(ROTOR *rot,float val)
{
  float rot_degr;      // current pos. rotor in decdegrees
  float req_degr;      // requested pos. rotor in decdegrees
  float diff_degr;
  boolean busy;
  if (!rot) return 0;

  rot->req_degr=val;                      // requested degrees
  rot->degr=to_degr(rot);

  #if ROTORTYPE==ROTORTYPE_AE
    #if FULLRANGE_AZIM == false     // range azimut=0...+180
      if (rot->req_degr>270) rot->req_degr-=360;
    #else                           // range azimut=0...360
      if ((rot->rot_degr > 270) && (rot->req_degr+rot->round*360 < 90)) rot->round++;
      if ((rot->req_degr > 270) && (rot->rot_degr+rot->round*360 < 90)) rot->round--;
      rot->req_degr+=rot->round*360;
    #endif
  #endif
  busy=run_motor_soft_dd(rot);

  if ((rot->id==AX_ID) && (!rot->calibrating)) // ???? tedoen??? && (wifi_connected))
  {
    float dif=rot->req_degr-rot->degr;
    dspbuffer(0,"%c %5.1f%c%5.1f",AX_NAME[0],rot->req_degr,(dif<=0? '+' : '-'),abs(dif));
    dspprintf(0,0,dspbuffer(14,"%c",(rot->calibrated? 'c' : 'u')));
  }
  if ((rot->id==EY_ID) && (!rot->calibrating))
  {
    float dif=rot->req_degr-rot->degr;
    dspbuffer(0,"%c %5.1f%c%5.1f",EY_NAME[0],rot->req_degr,(dif<=0? '+' : '-'),abs(dif));
    dspprintf(0,1,dspbuffer(14,"%c",(rot->calibrated? 'c' : 'u')));
  }
  return busy;
}

// reset rotor to pos=steps
void reset_to_pos(ROTOR *rot,long step)
{
  if (!rot) return;
  #if SWAP_DIR
  step*=-1;
  #endif
  rot->rotated=step;
  #if MOTORTYPE == MOT_STEPPER
    CMDP(rot,setCurrentPosition(step));
  #endif
}

#define ABS(a) ((a)<0? (a)*-1 : (a))
boolean in_stormpos(ROTOR *ax,ROTOR*ey)
{
  if (((!ax) || (ABS(step2degr(ax,ax->rotated)-AX_STORM) < 0.5)) && ((!ey) || (ABS(step2degr(ey,ey->rotated)-EY_STORM)<0.5)))
    return true;
  return false;
}

float in_prod(ROTOR *ax,ROTOR *ey)
{
  double ax1,ax2,ey1,ey2;
  double x1,y1,z1,x2,y2,z2,u;
  float phi;

  ax1=D2R(ax->req_degr);
  ey1=D2R(ey->req_degr);
  ax2=D2R(ax->degr);
  ey2=D2R(ey->degr);

  z1=sin(ey1);
  x1=cos(ax1)*cos(ey1);
  y1=sin(ax1)*cos(ey1);

  z2=sin(ey2);
  x2=cos(ax2)*cos(ey2);
  y2=sin(ax2)*cos(ey2);

  u=(x1*x2+y1*y2+z1*z2); if (u>1) u=1; if (u<-1) u=-1;
  phi=acos(u);
  return R2D(phi);
}
