/*******************************************************************
 * RCSId: $Id: rotorctrl.ino,v 1.6 2025/08/04 15:19:30 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content:
 *   main program, with setup() and loop()
 *
 * functions:
 *   void setup(void)
 *   void loop(void)
 *
 * History: 
 *   
 * $Log: rotorctrl.ino,v $
 * Revision 1.6  2025/08/04 15:19:30  ralblas
 * _
 *
 * Revision 1.5  2025/08/03 16:37:06  ralblas
 * _
 *
 * Revision 1.4  2025/08/02 11:59:09  ralblas
 * _
 *
 * Revision 1.3  2025/06/18 12:54:48  ralblas
 * _
 *
 * Revision 1.1  2023/09/07 07:11:58  ralblas
 * Initial revision
 *
 * Revision 1.4  2023/09/07 07:11:58  ralblas
 * _
 *
 * Revision 1.3  2023/09/02 09:58:15  ralblas
 * _
 *
 * Revision 1.2  2023/07/25 07:48:43  ralblas
 * _
 *
 * Revision 1.1  2023/07/18 12:11:49  ralblas
 * Initial revision
 *
 * Revision 1.1  2021/07/29 08:18:30  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#include <stdio.h>
#include <string.h>
#include "rotorctrl.h"

// Connectivity
#if USE_WIFI
  #include <WiFi.h>
  WiFiServer Server(ServerPort);
  WiFiClient RemoteClient;
  #if ADD_OTA_UPLOAD
    #include <AsyncTCP.h>
    #include <AsyncElegantOTA.h>
    AsyncWebServer otaserver(80);
  #endif

  #if USE_SGP4
    KEPLER kepler;
    EPOINT refpos;
  #endif
#endif

#if USE_DISPLAY == DISPL_LCD
  #include <LiquidCrystal_I2C.h>
  LiquidCrystal_I2C lcd(0x27, 2, 1, 0, 4, 5, 6, 7, 3, POSITIVE);
#endif


#if MOTORTYPE == MOT_STEPPER       // stepper motor
  #include <AccelStepper.h>        // http://www.airspayce.com/mikem/arduino/AccelStepper/index.html
  AccelStepper stepperAX(1, PIN_ROTPWM_AX, PIN_ROTDIR_AX);
  AccelStepper stepperEY(1, PIN_ROTPWM_EY, PIN_ROTDIR_EY);

  #define CMD(n,r) ((AccelStepper *)(n.stepper))->r
  #define CMDP(n,r) ((AccelStepper *)(n->stepper))->r
#endif

// Global variables
ROTORPREFS rotprefs;
ROTOR gAX_rot,gEY_rot;
ROTOR *SAX_rot,*SEY_rot;
COMMANDS command;
int used_interface;
float phi;

// Pulse counter
static void pos_handler(ROTOR *rot)
{
  if (!rot) return;
  if (rot->dir)
  {
       rot->rotated++;
  }
  else
  {
    rot->rotated--;
  }
}

// Pulse counter rotor Azimut/X
#if PIN_ROTPLS_AX && PIN_ROTPLS_AX >=0
  static ISR_FUNC AX_pos_handler(void)
  {
    static unsigned long t;
    unsigned long tnu=millis();

    if (tnu-t > rotprefs.bounce_period) 
    {
      t=tnu;
      pos_handler(&gAX_rot);
    }
  }
#endif

// Pulse counter rotor Elevation/Y
#if PIN_ROTPLS_EY && PIN_ROTPLS_EY >=0
  static ISR_FUNC EY_pos_handler(void)
  {
    static unsigned long t;
    unsigned long tnu=millis();

    if (tnu-t > rotprefs.bounce_period) 
    {
      t=tnu;
      pos_handler(&gEY_rot);
    }
  }
#endif

// setup: preferences, display, WiFi
void setup(void)
{
  int err = 0;
  SAX_rot = NULL;
  SEY_rot = NULL;
  used_interface=0;

  // define serial input, if USB connected: for commands (if no wifi), or debugging
  Serial.begin(SERIAL_SPEED);       // Start Serial Communication Interface

  #if USE_PREFS
    load_prefs();
  #else
    setup_prefs();
  #endif

  #ifndef LED_BUILTIN
    #define LED_BUILTIN 2
  #endif
  pinMode(LED_BUILTIN, OUTPUT);     // used for simple calibration status

  memset(&command,0,sizeof(command));
  command.gotoval.east_pass=true;

  // --------- Displays ---------
  #if USE_DISPLAY == DISPL_LCD
    // define LCD display
    lcd.begin(20, 4);
    lcd.clear();
  #elif USE_DISPLAY == DISPL_OLED
     setup_oled();
  #endif
  dspprintf(0,0,"Setup");

  // --------- Rotors ---------
  #if ROTOR_AX
    SAX_rot = &gAX_rot;
  #endif
  #if ROTOR_EY
    SEY_rot = &gEY_rot;
  #endif

  setup_ax(SAX_rot);
  setup_ey(SEY_rot);

  // define interrupts for backpulsecounters
  #if PIN_ROTPLS_AX && PIN_ROTPLS_AX >=0
    attachInterrupt(digitalPinToInterrupt(PIN_ROTPLS_AX) , AX_pos_handler , PULSE_DETECT);
  #endif
  #if PIN_ROTPLS_EY && PIN_ROTPLS_EY >=0
    attachInterrupt(digitalPinToInterrupt(PIN_ROTPLS_EY) , EY_pos_handler , PULSE_DETECT);
  #endif

  // --------- Stepper motors ---------
  #if MOTORTYPE == MOT_STEPPER
    #ifdef PIN_AXEYEnable
      if (PIN_AXEYEnable >= 0)
        digitalWrite(PIN_AXEYEnable, HIGH);             // Enable use of M415C controllers
    #endif
  #endif

  // --------- WiFI ---------
  #if USE_WIFI
    if (WiFi.status() == WL_CONNECTED)
    { // reconnect wifi, so disconnect if it was still connected
      // Should never happen...???
      blink(5,100);
      disconnect_wifi();
    }

    // Connect wifi: accesspoint or normal
    // No WIFI connection: use accesspoint to prevent hanging in wifi connection
    if ((PIN_SW1>=0) && (digitalRead(PIN_SW1)==0))
    {
      xprintf(ct_serial,"ACCESSPOINT");
      used_interface=2;
      connect_wifi_ap(rotprefs.my_ssid2.c_str(), rotprefs.my_pswd2.c_str());
    }
    else
    {
      used_interface=1;
      const char *hnam=rotprefs.my_hnam1.c_str();
      if ((hnam)&&(strlen(hnam)>1)) WiFi.setHostname(rotprefs.my_hnam1.c_str());
      connect_wifi(rotprefs.my_ssid1.c_str(), rotprefs.my_pswd1.c_str());
    }

    Server.begin();
    #if USE_SGP4
      // get time if track needs to be calculated
      configTime((long)0,(int)0, NTPSERVER);
      get_ntp();

      // load defaults
      load_default_refpos(&refpos);
      load_default_kepler(&kepler); // just some defaults, to make keplerdata valid

      calc_sgp4_const(&kepler,true);
    #endif
  #endif

  blink(2,200);                     // indicate start setup

  #if USE_WIFI
    #if ADD_OTA_UPLOAD
      otaserver.on("/", HTTP_GET, [](AsyncWebServerRequest *request) {
        request->send(200, "text/plain", "Upload ESP32 firmware for rotor. Add '/update' to URL.");
      });
      AsyncElegantOTA.begin(&otaserver);    // Start AsyncElegantOTA
      otaserver.begin();
      Serial.println("HTTP server started");
    #endif
  #endif

  digitalWrite(LED_BUILTIN, LOW);   // LED off; start calibration
  delay(1000);

  // --------- Setup Calibration ---------
  if (SAX_rot)
  {
    SAX_rot->cal_progress=cal_notdone;
    command.gotoval.ax = SAX_rot->degr;
  }
  if (SEY_rot)
  {
    SEY_rot->cal_progress=cal_notdone;
    command.gotoval.ey = SEY_rot->degr;
  }

  // If SW2=0 then skip calibration at power-up.
  if ((PIN_SW2>=0) && (digitalRead(PIN_SW2)==0))
  {
    if (SAX_rot)
    {
      SAX_rot->cal_progress=cal_err;  // sets CAL_READY to true, disabling cal. (see first commands loop)s
      set_led(SAX_rot,7,1);           // set white: don't calibrate
    }
    if (SEY_rot)
    {
      SEY_rot->cal_progress=cal_err;
      set_led(SEY_rot,7,1);           // set white: don't calibrate
    }
  }

  dspprintf(0,0,"End Setup!");
  #if USE_WIFI
  if (used_interface==1)
  {
    byte *bssid=WiFi.BSSID();
    const char *ip=WiFi.localIP().toString().c_str();
  
    if (ip) dspprintf(0,2,"%s",ip+3);
    dspprintf(0,3,"%d dBm @ %02X%02X",WiFi.RSSI(),bssid[0],bssid[1]);
  }
  else
  {
    dspprintf(0,3,"Accesspoint");
  }
  #endif
//  dspprintf(0,2,"Interf: %s",(used_interface==1? "wifi":used_interface==2?"wifi ap" : "usb"));
  delay(3000);
}

#define CAL_READY(rot) ((!rot) || ((rot->cal_progress==cal_ready) || (rot->cal_progress==cal_err)))
#define CAL_ERR(rot)   ((rot) && (rot->cal_progress==cal_err))

// endless loop: catch position from serial interface and run motors
void loop(void)
{
  int speed[]={rotprefs.spd_cal1,rotprefs.spd_cal2};
  static boolean calibrating;
  // --------- Calibration ---------
  if (!CAL_READY(SAX_rot))
  {
    #if CAL_ZENITH
      calrot_zenit(SAX_rot);
    #else
      calrot_estop(SAX_rot,speed);
    #endif
    calmon(SAX_rot);
    calibrating=true;
  }

  if (!CAL_READY(SEY_rot))
  {

    #if CAL_ZENITH
      calrot_zenit(SEY_rot);
    #else
      calrot_estop(SEY_rot,speed);
    #endif
    calmon(SEY_rot);
    calibrating=true;
  }

  if (calibrating) send_posx(SAX_rot,SEY_rot);  // send calibration info back to monitor

  if ((CAL_READY(SAX_rot)) && (CAL_READY(SEY_rot)))
  { 
    if  ((CAL_ERR(SAX_rot)) || (CAL_ERR(SEY_rot)))
    {
      if (calibrating)
      {
        blink(10, 100);       // Note: causes pin LED_BUILTIN to pulse!
      }
      calibrating=false;
    }
    else
    {
      calibrating=false;
    }
  }

  if (calibrating)
  {
    if (SAX_rot) dspprintf(0,0,"AX d=%.1f",SAX_rot->degr);
    if (SAX_rot) dspprintf(0,1,"AX r=%d"  ,SAX_rot->rotated);
    if (SEY_rot) dspprintf(0,2,"EY d=%.1f",SEY_rot->degr);
    if (SEY_rot) dspprintf(0,3,"EY r=%d"  ,SEY_rot->rotated);
  }


  // --------- Read commands ---------
  if (Serial.available())
  {
    readCommand_serial();        // from USB, do command
  }

  #if USE_WIFI
    if ((WiFi.status() == WL_CONNECTED) || (used_interface==2))
    {
      bool rd_ok;
      rd_ok=readCommand_wifi();

      if (!calibrating)
      {
        if (rd_ok)
        {
          dspprintf(0,2,"%s err=%.1f",(SEY_rot->flipped? "West" : "East"),phi);
        }
        else
        {
          dspprintf(0,2,"NO CONTROL    ");
        }
        if (used_interface==1)
        {
          byte *bssid=WiFi.BSSID();
          dspprintf(0,3,"%d dBm @ %02X%02X",WiFi.RSSI(),bssid[0],bssid[1]);
        }
        else
        {
          dspprintf(0,3,"Accesspoint");
        }
      }
    }
    else
    {
      if (!calibrating)
      {
        dspprintf(0,2,"NO CONTROL    ");
        dspprintf(0,3,"NO WIFI CONN. ");
      }
    }
  #endif

  // Next is to keep requested pos. in line with calibration, so at end calibration rotors will stop.
  if ((!CAL_READY(SAX_rot)) || (!CAL_READY(SEY_rot)))
  {
    if (SAX_rot) command.gotoval.ax = SAX_rot->degr;
    if (SEY_rot) command.gotoval.ey = SEY_rot->degr;
    return;                           // busy calibrating
  }

  // Here: calibration ready or not done

  #if USE_SGP4
    if (command.run_calc)
    {
      calc_pos(&command.gotoval,&kepler,&refpos);
      
      #ifdef CONTSENDINFO
      {
        static int pa,pe;
        // detect >=1 degrees step
        if ((int)(command.gotoval.a)!=pa) command.got_new_pos=true;
        if ((int)(command.gotoval.e)!=pe) command.got_new_pos=true;
        pa=(int)(command.gotoval.a);
        pe=(int)(command.gotoval.e);
      }
      #endif
    }
  #endif

  if (command.contrunning)
  { // especially needed for stepper motors, see spec 'AccelStepper'
    run_motor_hard(SAX_rot, command.a_spd);
    run_motor_hard(SEY_rot, command.b_spd);
  }
  else
  {
    GOTO_VAL gvo=command.gotoval;

    #if defined(USE_EASTWEST) && USE_EASTWEST
      boolean flipped;
      flipped=convert_eastwest(&command.gotoval,&gvo);
      if (SAX_rot) SAX_rot->flipped=flipped;
      if (SEY_rot) SEY_rot->flipped=flipped;
    #endif

    rotor_goto(SAX_rot, gvo.ax);
    rotor_goto(SEY_rot, gvo.ey);
    phi=in_prod(SAX_rot,SEY_rot);

    #ifdef CONTSENDINFO
     #if USE_WIFI
      // if pos. is calculated in controller then send results back for monitoring 
      if (command.got_new_pos)
      {
        send_ctrldata(RemoteClient,SAX_rot, SEY_rot,&gvo); // send pos. to PC for monitoring
        usleep(1000000);
      }
     #endif
    #endif
    command.got_new_pos = false;
  }

  digitalWrite(LED_BUILTIN, in_stormpos(SAX_rot,SEY_rot));   // LED on: rotors in storm position
}
