/*******************************************************************
 * RCSId: $Id: handle_commands.ino,v 1.3 2025/06/19 09:59:29 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content: 
 *   handling of commands
 *
 * public functions:
 *   int parse_cmd(char *cmd)
 *      parses command in cmd and fills global 'command'
 *      returns 1 if command was processed
 *
 *   boolean execute_cmd(boolean)
 *      executes commands in global 'command'
 *
 * History: 
 *   
 * $Log: handle_commands.ino,v $
 * Revision 1.3  2025/06/19 09:59:29  ralblas
 * _
 *
 * Revision 1.2  2025/05/06 17:26:54  ralblas
 * _
 *
 * Revision 1.1  2025/03/25 19:55:52  ralblas
 * Initial revision
 *
 * Revision 1.2  2023/09/06 18:51:12  ralblas
 * _
 *
 * Revision 1.1  2023/07/18 17:52:14  ralblas
 * Initial revision
 *
 * Revision 1.1  2021/07/29 08:27:25  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#if USE_SGP4
  extern KEPLER kepler;
  extern EPOINT refpos;
#endif

#if USE_PREFS
  extern ROTORPREFS rotprefs;
#endif

extern ROTOR *SAX_rot,*SEY_rot;
extern COMMANDS command;
extern int used_interface;

// Get arg. from command
//   cmd: command-string:
//     <cmd> <val(s)> or <cmd>=<val(s)>, <val(s)>=string without spaces
//   key: command
//   return: string with value or NULL (key doesn't match)
//
#define LENBUF 50
static char *get_val(char *cmd,char *key)
{
  char *p;
  static char cmdi[LENBUF];
  strncpy(cmdi,cmd,LENBUF);
  if ((p=strchr(cmdi,' '))) *p='=';    // for backward compatibility
  if ((p=strchr(cmdi,'=')))            // <key>=<val>
  {
    if (!strncmp(cmdi,key,strlen(key))) return p+1;
  }
  return NULL;
}

// Get boolean from command
//   cmd: command-string:
//     <cmd> <bool>, <bool>=0  or 1
//   key: command
//   val: found boolean
//   return: 1 or 0 (key doesn't match)
static int get_bool(char *cmd,char *key,boolean *val)
{
  char *p;
  if ((p=get_val(cmd,key)))
  {
    if (atoi(p)) *val=true; else *val=false;
    return 1;
  }
  return 0;
}

#if USE_SGP4
// Set if communicated keplers are in degrees or radians (communicated = what's exchanged between pc and rotorctrl)
#define kepler_in_degrees true // temp., MUST be true with version >=2023.3! 
void convert_kep(float ival,float *roval,float *doval)
{
  if (kepler_in_degrees)
  {
    *doval=ival;
    *roval=D2R(ival);
  }
  else
  {
    *roval=ival;
    *doval=R2D(ival);
  }
}

// get Kepler from interface
static int get_kepler_item(char *cmd,KEPLER *k)
{
  char *p;
  if      ((p=get_val(cmd,"satname=")))      strncpy(k->name,p,20);
  else if ((p=get_val(cmd,"epoch_year=")))   k->epoch_year=atoi(p);
  else if ((p=get_val(cmd,"epoch_day=")))    k->epoch_day=atof(p);
  else if ((p=get_val(cmd,"decay_rate=")))   k->decay_rate=atof(p);
  else if ((p=get_val(cmd,"bstar=")))        k->bstar=atof(p);
  else if ((p=get_val(cmd,"inclination=")))  convert_kep(atof(p),&k->inclination,&k->d_inclination);
  else if ((p=get_val(cmd,"raan=")))         convert_kep(atof(p),&k->raan,&k->d_raan);
  else if ((p=get_val(cmd,"eccentricity="))) k->eccentricity=atof(p);
  else if ((p=get_val(cmd,"perigee=")))      convert_kep(atof(p),&k->perigee,&k->d_perigee);
  else if ((p=get_val(cmd,"anomaly=")))      convert_kep(atof(p),&k->anomaly,&k->d_anomaly);
  else if ((p=get_val(cmd,"motion=")))       k->motion=atof(p);
  else  return 0;
//  calc_sgp4_const(k,false);
//  get_ntp();                 // get fresh time
  return 1;
}
#endif

// parsing commands from serial or wifi interface
// format commands: (no spaces!)
//    <command>=<val>\n
//    <command> <val>\n (for backwards compatibility, only place with space!)
//    <command>\n
// !!! command MUST end with '\n'!!!
int parse_cmd(char *cmd)
{
  char *p;
  if ((p=strchr(cmd,'\n'))) *p=0;
  if ((p=strchr(cmd,'\r'))) *p=0;
  if ((p=get_val(cmd,"a=")))
  {
    command.contrunning=true;
    command.cmd=contrun_ax;
    command.a_spd=atoi(p);
    return 1;
  }
  else if ((p=get_val(cmd,"b=")))
  {
    command.contrunning=true;
    command.cmd=contrun_ey;
    command.b_spd=atoi(p);
    return 1;
  }
  else if (strlen(cmd))         // no 'a', 'b', but another command
  {
    command.contrunning=false;  // stop contrunning mode
  }

  if (!strcmp(cmd,"request_id"))
  {
    command.cmd=send_id;
    return 1;
  }
  if (!strcmp(cmd,"version"))
  {
    command.cmd=send_version;
    return 1;
  }

  if (!strcmp(cmd,"gc"))
  {
digitalWrite(LED_BUILTIN, 1);
    command.cmd=config;
    return 1;
  }
  if (!strcmp(cmd,"gs"))
  {
    command.cmd=status;
    return 1;
  }
  if ((p=get_val(cmd,"m=")))
  {
    command.cmd=monitor;
    return 1;
  }
  if (!strcmp(cmd,"cal"))
  {
    command.cmd=do_calibrate;
    return 1;
  }
  if (!strcmp(cmd,"storm"))
  {
    command.cmd=do_gotostorm;
    command.gotoval.ax=AX_STORM;
    command.gotoval.ey=EY_STORM;
    return 1;
  }
  if ((p=get_val(cmd,"f=")))
  {
    command.cmd=pwm_freq;
    command.pwm_freq=atoi(p);
    return 1;
  }

  if (!strcmp(cmd,"setup"))
  {
    command.cmd=do_setup;
    return 1;
  }
  if (!strcmp(cmd,"restart"))
  {
    command.cmd=restart;
    return 1;
  }

  if (get_bool(cmd,"pri_cmd=",&command.pri_cmd))
  {
    return 1;
  }

  if (!strcmp(cmd,"download_wifiinfo"))    // download to PC
  {
    command.cmd=send_wifiinfo;
    return 1; 
  }

  #if USE_PREFS
  if (!strcmp(cmd,"get_prefs"))      // download to PC
  {
    command.cmd=do_send_prefs;
    return 1; 
  }

  {
    char *pcmd=cmd;
    bool force=false;
    if (cmd[0]=='!')
    {
      force=true;
      pcmd++;
    }

    // cmd="PREF: <item>=<val>
    // e.g.: cmd="PREF: AX_MINSPEED=50"
    if (!strncmp(pcmd,"PREF: ",strlen("PREF: ")))  // upload from PC
    {
      get_prefs(pcmd+strlen("PREF: "),used_interface,force);
      setup_prefs_ax(SAX_rot);
      setup_prefs_ey(SEY_rot);

      return 1; 
    }

    // cmd="SAVE: <item>
    // e.g.: cmd="SAVE: AX_MINSPEED"
    // e.g.: cmd="SAVE: ALL"
    if (!strncmp(pcmd,"SAVE: ",strlen("SAVE: ")))  // save preferences
    {
      save_prefs(pcmd+strlen("SAVE: "),used_interface,force);
      return 1; 
    }
  }
  #endif

  #if USE_SGP4
    if (get_bool(cmd,"run_calc=",&command.run_calc))
    {
      if (command.run_calc)
        get_ntp();                         // get fresh time
        calc_sgp4_const(&kepler,kepler_in_degrees);

      return 1;
    }

    if ((p=get_val(cmd,"download_time")))  // download time to PC
    {
      command.cmd=send_time;
      return 1;
    }

    if ((p=get_val(cmd,"upload_time=")))   // upload time from PC
    {
      command.cmd=get_time;
      strncpy(command.time,p,sizeof(command.time));
      return 1;
    }

    if (!strcmp(cmd,"download_refpos"))    // download to PC
    {
      command.cmd=send_refpos;
      return 1; 
    }


    if ((p=get_val(cmd,"upload_refpos="))) // upload from PC
    {
      command.cmd=get_refpos;
      command.ref_lat=atof(p);
      if (p=strchr(p,',')) command.ref_lon=atof(p+1);
      return 1;
    }

    if (!strcmp(cmd,"download_keplers"))   // download to PC
    {
      command.cmd=send_kep;
      return 1; 
    }

    if (!strcmp(cmd,"get_ctrldata"))
    {
      command.cmd=send_rotdata;
      return 1; 
    }

    if (!strcmp(cmd,"get_pos"))
    {
      command.cmd=send_satpos;
      return 1; 
    }

    if (get_kepler_item(cmd,&kepler))
    {
      return 1; // get one kepler-item
    }

  #endif


  // last to check! Rest is for direct rotor commands.
  // If only numbers: change into gotopos command
  if ((isdigit(cmd[0])) && (strchr(cmd,','))) // command <val1>,<val2>[,<val3>]
  {
    char cmd2[30];
    strncpy(cmd2,cmd,30);
    sprintf(cmd,"gotopos=%s",cmd2);
  }

  // gotopos=<ew>,<ax>,<ey> or <ax>,<ey>
  if ((p=get_val(cmd,"gotopos=")))
  {
    char *p1;
    int nrval=1;
    for (p1=p; *p1; p1++) if (*p1==',') nrval++;
    if (nrval>0) command.cmd=do_gotoval;

    command.gotoval.east_pass=1;
    if (nrval==3)                           // <ew>,<ax>,<ey>
    {
      command.gotoval.east_pass=atoi(p);    // <ew>
      if (!(p=strchr(p,','))) return 0;
      p++;
      nrval--;                              // remaining <ax>,<ey>
    }
    if (nrval==2)                           // <ax>,<ey>
    {
      command.gotoval.ax=atof(p);           // <ax>
      if (!(p=strchr(p+1,','))) return 0;
      p++;
      command.gotoval.ey=atof(p);           // <ey>
      nrval-=2;                             // ready
    }
    if (nrval==1)                           // only <ax> ( 1-rotor/motor system?)
    {
      command.gotoval.ax=atof(p);           // <ax>
      command.gotoval.ey=0.;                // no ey, set to 0
      nrval--;
    }
    return 1;
  }
  return 0;
}

// execute commands, done after parsing (parce_cmd())
// from_wifi=TRUE: command from wifi, otherwise from serial (USB)
boolean execute_cmd(COMTYPE ct)
{
  boolean valid_cmd=true;
  #if PROCESSOR == PROC_ESP
    if (command.cmd==restart)    ESP.restart();
    if (command.cmd==do_setup)   setup();
  #else
    if (command.cmd==restart)    setup();
    if (command.cmd==do_setup)   setup();
  #endif
  if (command.cmd==contrun_ax)   run_motor_hard(SAX_rot, command.a_spd);
  else if (command.cmd==contrun_ey)   run_motor_hard(SEY_rot, command.b_spd);
  else if (command.cmd==send_id)      xprintf(ct,"ID: rotorctrl\n");
  else if (command.cmd==send_version) xprintf(ct,"VERS: Release %s\n",RELEASE);

  else if (command.cmd==config)       send_specs(SAX_rot, SEY_rot);
  else if (command.cmd==status)       send_stat(ct,SAX_rot, SEY_rot);

  else if (command.cmd==do_calibrate)
  {
    command.a_spd=0;
    command.b_spd=0;
    // Next commands enable calibration done in loop()
    SAX_rot->calibrated=false; SAX_rot->cal_progress=cal_notdone;
    SEY_rot->calibrated=false; SEY_rot->cal_progress=cal_notdone;
  }

  else if (command.cmd==do_gotostorm)
  {
    command.contrunning=false;
    command.got_new_pos=true;
  }

  else if (command.cmd==do_gotoval)
  {
    command.contrunning=false;
    command.got_new_pos=true;
  }

  #if USE_PREFS
    else if (command.cmd==do_send_prefs) send_prefs(ct,&rotprefs);
  #endif

  #if USE_WIFI
    else if (command.cmd==get_time)
    {
      set_time(command.time);
    }

    else if (command.cmd==send_time)
    {
      send_ctrltime(RemoteClient);
    }
    else if (command.cmd==send_wifiinfo)
    {
      do_send_wifiinfo(RemoteClient);
//      xprintf("WIFI=IP: %s PWR: %d\n",WiFi.localIP().toString().c_str(),WiFi.RSSI());
    }

  #endif

  #if USE_SGP4
    else if (command.cmd==send_rotdata)
    {
      send_ctrldata(RemoteClient,SAX_rot,SEY_rot,&command.gotoval);
    }
    else if (command.cmd==send_satpos)
    {
      send_pos(RemoteClient,&command.gotoval,kepler.name);
    }

    else if (command.cmd==send_refpos)
    {
      send_refposition(RemoteClient,&refpos);
    }

    else if (command.cmd==get_refpos)
    {
      refpos.lat=D2R(command.ref_lat);
      refpos.lon=D2R(command.ref_lon);
    }
    else if (command.cmd==send_kep)
    {
      send_keplers(RemoteClient,&kepler,kepler_in_degrees);
    }
  #endif
  else
    valid_cmd=false;
    

//  if (command.cmd!=none) printf("Command: %d\n",command.cmd);
  command.cmd=none;
  return valid_cmd;
}

