/*******************************************************************
 * RCSId: $Id: calibrate.ino,v 1.5 2025/08/06 08:31:31 ralblas Exp $
 *
 * Project: rotordrive
 * Author: R. Alblas
 *
 * content: 
 *   calibration of one or two motors
 *
 * public functions:
 *   void calibrate(ROTOR *AX_rot, ROTOR *EY_rot)
 *
 * History: 
 *   
 * $Log: calibrate.ino,v $
 * Revision 1.5  2025/08/06 08:31:31  ralblas
 * _
 *
 * Revision 1.4  2025/08/04 15:16:36  ralblas
 * _
 *
 * Revision 1.3  2025/08/03 11:16:24  ralblas
 * _
 *
 * Revision 1.2  2025/05/06 14:53:48  ralblas
 * _
 *
 * Revision 1.1  2023/09/05 17:36:31  ralblas
 * Initial revision
 *
 * Revision 1.6  2023/09/05 17:36:31  ralblas
 * _
 *
 * Revision 1.5  2023/07/25 09:28:07  ralblas
 * _
 *
 * Revision 1.4  2023/07/25 08:02:54  ralblas
 * _
 *
 * Revision 1.2  2023/07/21 19:06:18  ralblas
 * _
 *
 * Revision 1.1  2023/07/19 12:51:29  ralblas
 * Initial revision
 *
 *
 *******************************************************************/
/*******************************************************************
 * Copyright (C) 2020 R. Alblas. 
 *
 * This is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software. If not, write to the Free Software 
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 
 * 02111-1307, USA.
 ********************************************************************/
#include "rotorctrl.h"
extern ROTOR AX_rot,EY_rot;
extern COMMANDS command;

/*********************************************************************
 * Calibrate rotors
 *   run for some time both motors from start position
 *   run both motors back to start position = end switch
 *   run both motors 
 * return: 0 if OK, 1 if error
 * AX_rot/EY_rot have calibration status:
 *   0=not done yet
 *   1=start
 *   2=run forward
 *   3=stop forward
 *   4=run backwards to end-stop
 *   5=stop, run to offset  
 *   6=ready, set calibrating=false
 * If calibrating stays true: error; status shows which rotor gives error.
 * AX_rot or EY_rot may be NULL to calibrate a single rotor
 *********************************************************************/
extern ROTORPREFS rotprefs;

// set speed soft to go to zenith
static boolean set_zspeed(ROTOR *rot,int speed,int dir)
{
  boolean busy;
  int zen;
  if (!rot) return 0;
  if (dir) speed=speed*-1;
  zen=get_zenpos(rot);
  if (zen!=dir) speed=0;
  busy=run_motor_soft(rot,speed);
  return busy;
}

static void reset_for_cal(ROTOR *rot)
{
  boolean led_ena=true;
  if (!rot) return;
  set_led(rot,1,led_ena);              // set B: start cal.
  rot->calibrated=false;
  reset_to_pos(rot,0);
}

static int get_zenpos(ROTOR *rot)
{
  int zen;
  if (!rot) return -1;

  zen=digitalRead(rot->pin_zen);
  if (rot->zenpin_inv) zen=(zen? false : true);

  return zen;
}

#if CAL_ZENITH
int calrot_zenit(ROTOR *rot)
{
  int speed=rotprefs.spd_cal1;
  if (!rot) return 0;

  rot->calibrating=true;
  switch(rot->cal_progress)
  {
    case cal_notdone: default:
      run_motor_hard(rot,0);                  // stop motor
      command.gotoval.east_pass=true;         // keep rotors after calibration at 0, don't flip

      if (rwait(rot,1))
        rot->cal_progress=cal_prep_to_zenit;
    break;

    case cal_prep_to_zenit:                // prepare to run from switch:
      rot->zen_side=get_zenpos(rot);
      rot->cal_progress=goto_zenit;
    break;

    case goto_zenit: 
      if (!set_zspeed(rot,speed,rot->zen_side)) // run motor, stop if zenit passed
      {
        rot->zen_side=get_zenpos(rot);          // get zenit side
        
        if (rwait(rot,1))
        {
          if (rot->zen_side)                      // wrong side for calibration
          {
            rot->cal_progress=goto_zenit_reverse;
          }
          else
          {
            rot->cal_progress=cal_to_zenit;       // right side for calibrtaion
          }
        }
      }
    break;
    case goto_zenit_reverse:
      if (!set_zspeed(rot,speed,rot->zen_side)) // go through zenit side, to right size
      {
        rot->zen_side=get_zenpos(rot);
        
        if (rwait(rot,1))
          rot->cal_progress=cal_to_zenit;         // right position for calibration
      }
    break;
    case cal_to_zenit:                          // Do calibration: low speed through zenit
      rot->degr=rot->refpos;
      rot->rotated=from_degr(rot);
      if (!set_zspeed(rot,speed,rot->zen_side))
        rot->cal_progress=cal_ready;           // Ready.
    break;

    // 2 endpoints: rot->cal_progress=cal_err and cal_ready.
    case cal_err:
      run_motor_hard(rot,0);

      rot->calibrated=false;
      rot->calibrating=false;
      return 1;                               // end-of-calibration (though errored)
    break;

    case cal_ready:
      run_motor_hard(rot,0);                  // stop
      rot->degr=rot->refpos;                  // Set degrees to reference position
      rot->rotated=from_degr(rot);            // Set pulse count to reference position

      rot->calibrated=true;
      rot->calibrating=false;
      return 1;                               // end-of-calibration, OK
    break;
  }

}

#else
#define DEGR_FORWARD 5.
int calrot_estop(ROTOR *rot,int *ispeed)
{
  static float pos;
  boolean busy,moving;
  int speed[2];
  if (!rot) return 0;
  speed[0]=ispeed[0];
  speed[1]=ispeed[1];

  #if SWAP_DIR == false
    speed[0]*=-1;
    speed[1]*=-1;
  #endif

  rot->calibrating=true;

  switch(rot->cal_progress)
  {
    case cal_notdone: default:
      run_motor_hard(rot,0);
      command.gotoval.east_pass=true;           // keep rotors after calibration at 0, don't flip

      if (rwait(rot,1))
        rot->cal_progress=cal_prep_from_endpos; // wait after forced motorstop
      rot->calibrating=true;
    break;

    // A: Move from endswitch
    case cal_prep_from_endpos:                  // prepare to run from switch:
    {
      pos=DEGR_FORWARD;                         //   go DEGR_FORWARD degrees from end-switch
      reset_for_cal(rot);
      pos=pos+to_degr(rot);                     //   force to go forward DEGR_FORWARD degrees
      rot->timeout=millis();                    //   start time: for timeout

      rot->cal_progress=cal_from_endpos;
    }
    break;

    case cal_from_endpos:                       // run from end position
    {
      busy=rotor_goto(rot,pos);                 //   running, busy=0 if ready
      moving=is_moving(rot);                    //   detect if moving
      if ((!busy) || (!(moving)))
      {                                         //   ready, or rotor at other end, so didn't move? 
        run_motor_hard(rot,0);                  //   stop

        if (rwait(rot,1))
          rot->cal_progress=cal_prep_to_endpos;
      }
    }
    break;

    // B: Move to endswitch
    case cal_prep_to_endpos:                    // prepare to run to switch:
      reset_to_pos(rot,-1*rot->POffset);        //   set puls counter to # pulses to calibration position
      rot->timeout=millis();                    //   for timeout

      rot->cal_progress=cal_to_endpos;          // 
    break;

    case cal_to_endpos:                         // run to end switch
      run_motor_soft(rot,speed[0]);             //   run with speed 'speed'
      if (!(is_moving(rot)))                    //   detect moving
      {
        rot->cal_progress=cal_prep_to_calpos;   //   motor stopped, at end switch
      }
    break;

    // C: Move to calibration point
    case cal_prep_to_calpos:                    // prepare go to calibration point
      reset_to_pos(rot,-1*rot->POffset);        //   set such that rotor stops 'POffset' pulses after end-switch
      pos=0;
      rot->timeout=millis();                    //   for timeout

      rot->cal_progress=cal_to_calpos;          //
    break;

    case cal_to_calpos:                         // go to calibration point
      if (!rotor_goto(rot,pos))               //   stopped at desired position
      {
        rot->cal_progress=cal_ready;
        run_motor_hard(rot,0);                  // stop
        rot->degr=rot->refpos;                  // Set degrees to reference position
        rot->rotated=from_degr(rot);            // Set pulse count to reference position

        rot->calibrated=true;
        rot->calibrating=false;
        return 1;                               // end-of-calibration, OK
      }

      if (!(is_moving(rot)))                    //   timeout, stop, error
      {
        rot->cal_progress=cal_err;
        run_motor_hard(rot,0);

        rot->calibrated=false;
        rot->calibrating=false;
        return 1;                               // end-of-calibration (though errored)
      }
    break;

    // rest not entered ever...
    // D1: endpoint Errored
    case cal_err:
      run_motor_hard(rot,0);

      rot->calibrated=false;
      rot->calibrating=false;
      return 1;                               // end-of-calibration (though errored)
    break;

    // D2: endpoint OK
    case cal_ready:
      run_motor_hard(rot,0);                  // stop
      rot->degr=rot->refpos;                  // Set degrees to reference position
      rot->rotated=from_degr(rot);            // Set pulse count to reference position

      rot->calibrated=true;
      rot->calibrating=false;
      return 1;                               // end-of-calibration, OK
    break;
  }
  return 0;
}
#endif

void calmon(ROTOR *rot)
{
  int led_ena=1;
  if (!rot) return;
  switch(rot->cal_progress)
  {
    case cal_notdone:
      set_led(rot,1,led_ena);  // set B
    break;
    case cal_prep_from_endpos:
      set_led(rot,1,led_ena);  // set B
    break;
    case cal_from_endpos:
      set_led(rot,1,led_ena);  // set B
    break;
    case cal_prep_to_endpos:
      set_led(rot,5,led_ena);  // set RB
    break;
    case cal_to_endpos:
      set_led(rot,5,led_ena);  // set RB
    break;
    case cal_prep_to_calpos:
      set_led(rot,6,led_ena);  // set RG
    break;
    case cal_to_calpos:
      set_led(rot,6,led_ena);  // set RG
    break;
    case cal_ready:
      set_led(rot,2,led_ena);  // set G
    break;
    case cal_err:
      set_led(rot,4,led_ena);  // set R
    break;
  }
}

